package ru.yandex.direct.ansiblejuggler.model;

import java.util.ArrayList;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.annotation.JsonAutoDetect;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.databind.PropertyNamingStrategy;
import com.fasterxml.jackson.databind.annotation.JsonNaming;

import ru.yandex.direct.ansiblejuggler.PlaybookUtils;
import ru.yandex.direct.ansiblejuggler.model.checks.JugglerCheck;
import ru.yandex.direct.ansiblejuggler.model.tasks.JugglerCheckTask;
import ru.yandex.direct.ansiblejuggler.model.tasks.JugglerCleanupTask;
import ru.yandex.direct.ansiblejuggler.model.tasks.JugglerFactsTask;
import ru.yandex.direct.ansiblejuggler.model.tasks.JugglerTask;

/**
 * Описывает один play для ansible-playbook.
 *
 * @see JugglerCheck
 * @see JugglerPlaybook
 */
@JsonAutoDetect(getterVisibility = JsonAutoDetect.Visibility.NONE)
@JsonInclude(JsonInclude.Include.NON_EMPTY)
@JsonNaming(PropertyNamingStrategy.SnakeCaseStrategy.class)
@ParametersAreNonnullByDefault
public class JugglerPlay {
    public static final String DEFAULT_PLAYBOOK_HOST = "localhost";
    public static final String DEFAULT_PLAYBOOK_CONNECTION = "local";
    public static final boolean DEFAULT_PLAYBOOK_GATHERING_FACTS = false;

    @JsonProperty
    private final String hosts;

    @JsonProperty
    private final String connection;

    @JsonProperty
    private final boolean gatherFacts;

    @JsonProperty
    private final List<JugglerTask> preTasks;

    @JsonProperty
    private final List<JugglerTask> tasks;

    @JsonProperty
    private final List<JugglerTask> postTasks;

    /**
     * Создать пустой play cо следующими умолчаниями:
     * <ul>
     * <li>список хостов для применения - {@value DEFAULT_PLAYBOOK_HOST}</li>
     * <li>способ подключения - {@value DEFAULT_PLAYBOOK_CONNECTION}</li>
     * <li>включен ли сбор фактов - {@value DEFAULT_PLAYBOOK_GATHERING_FACTS}</li>
     * </ul>
     */
    public JugglerPlay() {
        hosts = DEFAULT_PLAYBOOK_HOST;
        connection = DEFAULT_PLAYBOOK_CONNECTION;
        gatherFacts = DEFAULT_PLAYBOOK_GATHERING_FACTS;

        preTasks = new ArrayList<>();
        tasks = new ArrayList<>();
        postTasks = new ArrayList<>();
    }

    /**
     * Создать play с умолчаниями и задачей по конфигурации ansible-juggler адресом API
     *
     * @param jugglerApiUrl juggler API url, например {@literal http://dimon.yandex.ru:8998/api}
     * @see JugglerPlay#JugglerPlay()
     */
    public JugglerPlay(String jugglerApiUrl) {
        this();
        withPreTask(new JugglerFactsTask(jugglerApiUrl));
    }

    /**
     * Получить playbook, состоящий из текущего play
     *
     * @return новый {@link JugglerPlaybook} состоящий из текущего play
     */
    public JugglerPlaybook asPlaybook() {
        return new JugglerPlaybook(this);
    }

    /**
     * Добавить в play задачу, которую требуется выполнить до основных задач
     *
     * @param task задача для добавления
     * @return текущий play
     */
    public JugglerPlay withPreTask(JugglerTask task) {
        preTasks.add(task);
        return this;
    }

    /**
     * Добавить в play задачу
     *
     * @param task задача для добавления
     * @return текущий play
     */
    public JugglerPlay withTask(JugglerTask task) {
        tasks.add(task);
        return this;
    }

    /**
     * Добавить в play задачу, которую требуется выполнить после основных задач
     *
     * @param task задача для добавления
     * @return текущий play
     */
    public JugglerPlay withPostTask(JugglerTask task) {
        postTasks.add(task);
        return this;
    }

    /**
     * Добавить в play одну проверку с указанием названия задачи
     *
     * @param taskName    название задачи (для улучшения читаемости плейбука)
     * @param checkParams параметры проверки
     * @return текущий play
     * @throws NullPointerException если любой из параметров равен {@code null}
     */
    public JugglerPlay withCheck(String taskName, JugglerCheck checkParams) {
        withTask(new JugglerCheckTask(taskName, checkParams));
        return this;
    }

    /**
     * Добавить в play одну проверку.
     *
     * @param checkParams параметры проверки
     * @return текущий play
     * @throws NullPointerException     если параметры равны {@code null}
     * @throws IllegalArgumentException при невалидном значении {@code checkParams}
     */
    public JugglerPlay withCheck(JugglerCheck checkParams) {
        withTask(new JugglerCheckTask(checkParams));
        return this;
    }

    /**
     * Добавить в play задачу по очистке проверок, отсутствующих в нем (по значению {@code jcheckMark}.
     * Удаляются проверки со всех упомянутых в play хостов, у которых метка совпадает с указанной
     * и при этом они не были описаны в нем
     *
     * @param jcheckMark метка для очистки, валидириуется {@link PlaybookUtils#checkJcheckMark(String)}
     * @return текущий play
     * @throws IllegalArgumentException при невалидном значении {@code jcheckMark}
     */
    public JugglerPlay withCleanup(String jcheckMark) {
        withPostTask(new JugglerCleanupTask(jcheckMark));
        return this;
    }

    /**
     * Получить количество задач в текущем play
     *
     * @return число задач
     */
    int getTasksCount() {
        return tasks.size();
    }
}
