package ru.yandex.direct.ansiblejuggler.model;

import java.io.File;
import java.io.IOException;
import java.util.Collections;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.core.JsonProcessingException;

import static ru.yandex.direct.ansiblejuggler.PlaybookUtils.getMapper;

/**
 * Описывает ansible-playbook с juggler-проверками.
 * <p>
 * Полученный плейбук можно применить с помощью {@link ru.yandex.direct.ansiblejuggler.AnsibleWrapper#syncPlaybook()}
 * Для этого потребуются:
 * <ul>
 * <li>ansible (версии 1.9.Х)</li>
 * <li>ansible-juggler </li>
 * </ul>
 *
 * @see <a href="http://docs.ansible.com/ansible/playbooks_intro.html#playbook-language-example">Playbook Example</a>
 * @see JugglerPlay
 * @see ru.yandex.direct.ansiblejuggler.AnsibleWrapper#AnsibleWrapper
 * @see <a href="https://wiki.yandex-team.ru/sm/juggler/ansible/#install">Установка ansible-juggler</a>
 */
@ParametersAreNonnullByDefault
public class JugglerPlaybook {
    private final List<JugglerPlay> playbook;

    /**
     * Сконструировать плейбук из одного play
     */
    JugglerPlaybook(JugglerPlay play) {
        playbook = Collections.singletonList(play);
    }

    /**
     * Получить текстовое представление плейбука в YAML-формате
     *
     * @return текст плейбука
     * @throws JsonProcessingException при ошибке сериализации
     */
    public String dump() throws JsonProcessingException {
        return getMapper().writeValueAsString(playbook);
    }

    /**
     * Сохранить плейбук в указанный файл
     *
     * @param file файл для сохранения
     * @throws IOException при ошибках сериализации или ввода/вывода
     */
    public void saveToFile(File file) throws IOException {
        getMapper().writeValue(file, playbook);
    }

    /**
     * Сохранить плейбук во временный файл
     *
     * @return новый {@link File}, в который был сохранен плейбук. файл будет удален по завершении JVM
     * @throws IOException при ошибках сериализации или ввода/вывода
     */
    public File saveToTempFile() throws IOException {
        File tempFile = File.createTempFile("juggler-checks-", ".yml");
        tempFile.deleteOnExit();
        saveToFile(tempFile);
        return tempFile;
    }

    /**
     * Получить количество задач внутри плейбука
     *
     * @return число задач (сумма по всем play)
     */
    public int getTasksCount() {
        return playbook.stream().mapToInt(JugglerPlay::getTasksCount).sum();
    }
}
