package ru.yandex.direct.ansiblejuggler.model;

import java.util.Objects;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.annotation.ParametersAreNonnullByDefault;

/**
 * Описывает резюме проверки или синхронизации плейбука для одного хоста.
 */
@ParametersAreNonnullByDefault
public class PlayRecap {
    public static class ParseException extends IllegalArgumentException {
        ParseException(String message) {
            super(message);
        }
    }

    private static final Pattern STAT_LINE_PATTERN = Pattern.compile(
            "[^:\\s]+\\s*:\\s*ok=([0-9]+)\\s*changed=([0-9]+)\\s*unreachable=([0-9]+)\\s*failed=([0-9]+)\\s*$");

    private final int ok;
    private final int changed;
    private final int unreachable;
    private final int failed;

    /**
     * Создать объект-рюзюме из строки статистики
     *
     * @param statLine строка статистики, выведенная ansible после {@literal PLAY RECAP *****}
     * @throws ParseException при ошибке парсинга строки
     */
    public PlayRecap(String statLine) throws ParseException {
        Matcher matcher = getStatMatcher(statLine);
        if (matcher.matches()) {
            ok = Integer.parseInt(matcher.group(1));
            changed = Integer.parseInt(matcher.group(2));
            unreachable = Integer.parseInt(matcher.group(3));
            failed = Integer.parseInt(matcher.group(4));
        } else {
            throw new ParseException("statLine doesn't match defined pattern");
        }
    }

    /**
     * Получить для строки её матчер на шаблон строки статистики
     *
     * @param line строка для анализа
     * @return {@code Matcher} объект для переданной строки
     */
    public static Matcher getStatMatcher(String line) {
        return STAT_LINE_PATTERN.matcher(line);
    }

    /**
     * @return количество успешных задач, которые не внесли изменений
     */
    public int getOk() {
        return ok;
    }

    /**
     * @return количество задач, внесших изменения
     */
    public int getChanged() {
        return changed;
    }

    /**
     * @return количество задач по недоступным хостам
     */
    public int getUnreachable() {
        return unreachable;
    }

    /**
     * @return количество проваленных задача
     */
    public int getFailed() {
        return failed;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        PlayRecap recap = (PlayRecap) o;
        return ok == recap.ok &&
                changed == recap.changed &&
                unreachable == recap.unreachable &&
                failed == recap.failed;
    }

    @Override
    public int hashCode() {
        return Objects.hash(ok, changed, unreachable, failed);
    }

    @Override
    public String toString() {
        return "ok=" + ok + ", changed=" + changed + ", unreachable=" + unreachable + ", failed=" + failed;
    }
}
