package ru.yandex.direct.ansiblejuggler.model.aggregators;

import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.annotation.JsonAutoDetect;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonProperty;

import ru.yandex.direct.ansiblejuggler.model.checks.JugglerChild;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkNotNull;

/**
 * Описывает аггрегатор и его параметры.
 * Конкретные их виды реализованы в виде классов-наследников.
 */
@JsonAutoDetect(getterVisibility = JsonAutoDetect.Visibility.NONE)
@JsonInclude(JsonInclude.Include.NON_EMPTY)
@ParametersAreNonnullByDefault
public abstract class JugglerAggregator {
    @JsonProperty(value = "aggregator")
    private final String name;

    @JsonProperty(value = "aggregator_kwargs")
    private AggregatorParams params;

    /**
     * Создать аггрегатор с заданными именем
     *
     * @param aggregatorName имя аггегатора в терминах juggler-сервера, одно из "Modules for aggregation"
     * @throws NullPointerException     если имя равно {@code null}
     * @throws IllegalArgumentException если имя пустое
     */
    JugglerAggregator(String aggregatorName) {
        checkNotNull(aggregatorName, "Aggregator name cannot be null");
        checkArgument(!aggregatorName.isEmpty(), "Aggregator name cannot be empty");
        name = aggregatorName;
    }

    /**
     * Задать в аггрегаторе "не учитывать сигналы с недоступных хостов".
     * Недоступность определяется по проверке {@code unreachableService}
     *
     * @param unreachableService проверка по которой juggler-сервер будет вычислять недоступность
     * @return текущий аггрегатор
     */
    public JugglerAggregator withSkipUnreachable(JugglerChild unreachableService) {
        getParams().withUnreachableParams(new UnreachableParams(ParamsMode.SKIP, unreachableService));
        return this;
    }

    /**
     * Задать в аггрегаторе "пропускать сигналы с хостов под downtime"
     *
     * @return текущий аггрегатор
     */
    public JugglerAggregator withSkipDowntimes() {
        return withDowntimesMode(ParamsMode.SKIP);
    }

    /**
     * Заменить в аггрегаторе параметры unreachable переданными
     *
     * @param unreachableParams как учитывать при аггрегации сигналы с недоступных хостов
     * @return текущий аггрегатор
     */
    public JugglerAggregator withUnreachableParams(UnreachableParams unreachableParams) {
        getParams().withUnreachableParams(unreachableParams);
        return this;
    }

    /**
     * Заменить в аггрегаторе параметры downtimes переданными
     *
     * @param downtimesMode как учитывать при аггрегации сигналы с хостов, по которым действует downtime
     * @return текущий аггрегатор
     * @see ParamsMode
     */
    public JugglerAggregator withDowntimesMode(ParamsMode downtimesMode) {
        getParams().withDowntimesMode(downtimesMode);
        return this;
    }

    /**
     * Параметры сделаны singleton'ом, чтобы пустые - не сериализовались
     */
    private AggregatorParams getParams() {
        if (params == null) {
            params = new AggregatorParams();
        }
        return params;
    }

    /**
     * Получить человекопонятное название аггрегатора
     *
     * @return строка с названием класса
     */
    public String getHumanReadableName() {
        return this.getClass().getSimpleName();
    }
}

