package ru.yandex.direct.ansiblejuggler.model.checks;

import java.time.Duration;
import java.util.Collection;
import java.util.LinkedHashSet;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.annotation.JsonAutoDetect;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.annotation.JsonUnwrapped;
import com.fasterxml.jackson.annotation.JsonValue;
import com.fasterxml.jackson.databind.PropertyNamingStrategy;
import com.fasterxml.jackson.databind.annotation.JsonNaming;

import ru.yandex.direct.ansiblejuggler.PlaybookUtils;
import ru.yandex.direct.ansiblejuggler.model.aggregators.JugglerAggregator;
import ru.yandex.direct.ansiblejuggler.model.meta.JugglerMeta;
import ru.yandex.direct.ansiblejuggler.model.notifications.JugglerNotification;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.base.Preconditions.checkState;
import static ru.yandex.direct.ansiblejuggler.PlaybookUtils.checkHost;
import static ru.yandex.direct.ansiblejuggler.PlaybookUtils.checkJcheckMark;
import static ru.yandex.direct.ansiblejuggler.PlaybookUtils.checkName;
import static ru.yandex.direct.ansiblejuggler.PlaybookUtils.checkTtl;

/**
 * Описывает juggler-проверку
 */
@JsonAutoDetect(isGetterVisibility = JsonAutoDetect.Visibility.NONE)
@JsonInclude(JsonInclude.Include.NON_EMPTY)
@JsonNaming(PropertyNamingStrategy.SnakeCaseStrategy.class)
@ParametersAreNonnullByDefault
public class JugglerCheck {
    private static class FlapParamsWrapper {
        private final FlapParams params;

        FlapParamsWrapper(@Nullable FlapParams params) {
            this.params = params;
        }

        @JsonValue
        private Object value() {
            if (params != null) {
                return params;
            } else {
                return Boolean.FALSE;
            }
        }
    }

    @JsonProperty
    private final String host;

    @JsonProperty
    private final String service;

    @JsonProperty
    private final Set<String> responsible;

    @JsonProperty
    private JugglerMeta meta;

    @JsonProperty
    private final Set<JugglerAlertMethod> alertMethod;

    @JsonProperty
    private final Set<JugglerNotification> notifications;

    @JsonProperty
    private final FlapParamsWrapper flap;

    @JsonProperty
    private final long ttl;

    @JsonProperty
    @JsonUnwrapped
    private final JugglerAggregator aggregator;

    @JsonProperty
    private final Set<JugglerChild> children;

    @JsonProperty
    private final Set<String> tags;

    @JsonProperty
    private final String namespace;

    @JsonProperty
    private final String jcheckMark;

    private boolean allowEmptyChildren;

    /**
     * Создать простую juggler-проверку
     *
     * @param targetHost    хост на который завести проверку (к нему будет относиться результат проверки)
     *                      валидируется {@link PlaybookUtils#checkHost(String)}
     * @param targetService имя проверки
     *                      валидируется {@link PlaybookUtils#checkName(String)}
     * @param ttl           TTL проверки
     *                      валидируется {@link PlaybookUtils#checkTtl(Duration)}
     * @param aggregator    аггрегатор для данной проверки
     * @param namespace     имя пространста имен проверки для ограничения доступа
     * @param jcheckMark    метка для проверки
     *                      валидириуется {@link PlaybookUtils#checkJcheckMark(String)}
     * @throws NullPointerException     если любой из переданных параметров равен {@code null}
     * @throws IllegalArgumentException при невалидных переданных параметрах
     */
    public JugglerCheck(String targetHost, String targetService, Duration ttl, JugglerAggregator aggregator,
                        String namespace, String jcheckMark) {
        this(targetHost, targetService, ttl, aggregator, namespace, jcheckMark, null);
    }

    /**
     * Создать juggler-проверку с возможность указать дополнительные параметры:
     * <ul>
     * <li>настройки флаподавилки</li>
     * <li>сделать проверку активной</li>
     * </ul>
     *
     * @param targetHost    хост на который завести проверку (к нему будет относиться результат проверки)
     *                      валидируется {@link PlaybookUtils#checkHost(String)}
     * @param targetService имя проверки
     *                      валидируется {@link PlaybookUtils#checkName(String)}
     * @param ttl           TTL проверки
     *                      валидируется {@link PlaybookUtils#checkTtl(Duration)}
     * @param aggregator    аггрегатор для данной проверки
     * @param namespace     имя пространста имен проверки для ограничения доступа
     * @param jcheckMark    метка для проверки
     *                      валидириуется {@link PlaybookUtils#checkJcheckMark(String)}
     * @param flap          параметры флаподавилки или {@code null}, если применять флаподав не нужно
     * @throws NullPointerException     если любой из переданных параметров (кроме последних двух) равен {@code null}
     * @throws IllegalArgumentException при невалидных переданных параметрах
     */
    public JugglerCheck(String targetHost, String targetService, Duration ttl, JugglerAggregator aggregator,
                        String namespace, String jcheckMark, @Nullable FlapParams flap) {
        checkHost(targetHost);
        checkName(targetService);
        checkTtl(ttl);
        checkNotNull(aggregator, "aggregator cannot be null");
        checkJcheckMark(jcheckMark);

        this.host = targetHost;
        this.service = targetService;
        this.responsible = new LinkedHashSet<>();
        this.alertMethod = new LinkedHashSet<>();
        this.notifications = new LinkedHashSet<>();
        this.flap = new FlapParamsWrapper(flap);
        this.ttl = ttl.getSeconds();
        this.aggregator = aggregator;
        this.children = new LinkedHashSet<>();
        this.tags = new LinkedHashSet<>();
        this.namespace = namespace;
        this.jcheckMark = jcheckMark;
    }

    /**
     * Добавить к проверке ответственного за {@code targetHost}.
     * Требуется только для случаев, когда хост заводится в големе при синхронизации плейбука
     * Желательно использовать существующие (в том числе виртуальные) хосты для проверок и не указывать ответственных
     *
     * @param responsible логин ответственного
     * @return текущая проверка
     * @throws NullPointerException
     * @throws IllegalArgumentException
     */
    @Deprecated
    public JugglerCheck withResponsible(String responsible) {
        checkNotNull(responsible, "responsible cannot be null");
        checkArgument(!this.responsible.contains(responsible), "check already contains this responsible");
        this.responsible.add(responsible);
        return this;
    }

    /**
     * Добавить к проверке ответственных за {@code targetHost}.
     * Требуется только для случаев, когда хост заводится в големе при синхронизации плейбука
     * Желательно использовать существующие (в том числе виртуальные) хосты для проверок и не указывать ответственных
     *
     * @param responsibles список логинов ответственных
     * @return текущая проверка
     * @throws NullPointerException
     * @throws IllegalArgumentException
     */
    @Deprecated
    public JugglerCheck withResponsibles(Collection<String> responsibles) {
        checkNotNull(responsibles, "responsibles cannot be null");
        checkArgument(!responsibles.isEmpty(), "responsibles shoud not be empty");
        responsibles.forEach(this::withResponsible);
        return this;
    }

    /**
     * Добавить к проверке метод уведовления
     *
     * @param method метод уведомления
     * @return текущая проверка
     * @throws NullPointerException     если указанный метод равен {@code null}
     * @throws IllegalArgumentException если указанный метод уже содержится в проверке
     */
    @Deprecated
    public JugglerCheck withAlertMethod(JugglerAlertMethod method) {
        checkNotNull(method, "method cannot be null");
        checkArgument(!this.alertMethod.contains(method), "check already contains this alert method");
        alertMethod.add(method);
        return this;
    }

    /**
     * Добавить к проверке мета-информацию
     *
     * @param meta мета-информация
     * @return текущая проверка
     */
    public JugglerCheck withMeta(JugglerMeta meta) {
        this.meta = meta;
        return this;
    }

    public JugglerMeta getMeta() {
        return meta;
    }

    /**
     * Добавить к проверке уведомление
     *
     * @param notification параметры уведомления
     * @return текущая проверка
     * @throws NullPointerException     если указанное уведомление равно {@code null}
     * @throws IllegalArgumentException если указанное уведомление уже содержится в проверке
     */
    public JugglerCheck withNotification(JugglerNotification notification) {
        checkNotNull(notification, "notification cannot be null");
        checkArgument(!this.notifications.contains(notification), "check already contains this notification");
        notifications.add(notification);
        return this;
    }

    /**
     * Дрбавить к проверке нотификации
     *
     * @param notifications параметры уведомлений
     * @return текущая проверка
     * @throws NullPointerException     если набор или уведомление равны {@code null}
     * @throws IllegalArgumentException если какое-либо уведомление уже содержится в проверке
     */
    public JugglerCheck withNotifications(Collection<JugglerNotification> notifications) {
        checkNotNull(notifications, "notifications cannot be null");
        notifications.forEach(this::withNotification);
        return this;
    }

    /**
     * Добавить к проверке дочерний элемент
     *
     * @param child дочерняя элемент (хост и сервис исходного события)
     * @return текущая проверка
     * @throws NullPointerException     если {@code child} равен {@code null}
     * @throws IllegalArgumentException если указанный элемент уже содержится среди дочерних
     */
    public JugglerCheck withChild(JugglerChild child) {
        checkNotNull(child, "child cannot be null");
        checkArgument(!this.children.contains(child), "check already contains this child");
        this.children.add(child);
        return this;
    }

    /**
     * Добавить к проверке дочерние элементы
     *
     * @param children коллекция дочерних элементов (хост+сервис исходных события)
     * @return текущая проверка
     * @throws NullPointerException     если колекция или любой ее элемент равны {@code null}
     * @throws IllegalArgumentException если коллекция пустая, или любой из элементов уже содержится среди дочерних
     */
    public JugglerCheck withChildren(Collection<JugglerChild> children) {
        checkNotNull(children, "children cannot be null");
        checkArgument(!children.isEmpty(), "children should not be empty");
        children.forEach(this::withChild);
        return this;
    }

    /**
     * Указать валидации, что "я точно знаю что делаю и хочу завести проверку без указания детей"
     *
     * @return текущая проверка
     */
    public JugglerCheck withNoChild() {
        allowEmptyChildren = true;
        return this;
    }

    /**
     * Добавить к проверке тег
     *
     * @param tag тег
     * @return текущая проверка
     * @throws NullPointerException     если тег {@code null}
     * @throws IllegalArgumentException если тег пустой
     * @throws IllegalStateException    если указанный тег уже содержится в проверке
     */
    public JugglerCheck withTag(String tag) {
        checkNotNull(tag, "tag cannot be null");
        checkArgument(!tag.isEmpty(), "tag cannot be empty");
        checkState(!this.tags.contains(tag), "check already contains this tag");
        this.tags.add(tag);
        return this;
    }

    /**
     * Добавить к проверке теги
     *
     * @param tags массив тегов
     * @return текущая проверка
     * @throws NullPointerException     если массив или его элементы равны {@code null}
     * @throws IllegalArgumentException если есть пустые теги
     * @throws IllegalStateException    если какой-либо тег уже содержится в проверке
     * @see #withTag(String)
     */
    public JugglerCheck withTags(Collection<String> tags) {
        checkNotNull(tags, "tags cannot be null");
        tags.forEach(this::withTag);
        return this;
    }

    /**
     * Является ли проверка валидной для добавления ее в плейбук
     */
    public boolean isValid() {
        return !children.isEmpty() || allowEmptyChildren;
    }

    @Override
    public String toString() {
        return host + ":" + service + " over " + children.size() + " children with " + aggregator.getHumanReadableName()
                + " aggregator";
    }
}
