package ru.yandex.direct.ansiblejuggler.model.checks;

import java.time.Duration;
import java.util.Collection;
import java.util.Collections;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.ansiblejuggler.model.aggregators.AnyChildShouldBeOk;

import static ru.yandex.direct.ansiblejuggler.PlaybookUtils.checkChildrenHost;
import static ru.yandex.direct.ansiblejuggler.PlaybookUtils.checkName;

/**
 * Обертка над {@link JugglerCheck} со следующими умолчаниями:
 * <ul>
 * <li>аггрегатор {@link AnyChildShouldBeOk}</li>
 * <li>не учитываются события с хостов под downtime</li>
 * <li>заполнен список дочерних проверок</li>
 * <li>флаподавилка выключена</li>
 * </ul>
 */
@ParametersAreNonnullByDefault
public class SingleEventFromAnyHostCheck extends JugglerCheck {

    private SingleEventFromAnyHostCheck(String host, String service, Duration ttl,
                                        String namespace, String jcheckMark) {
        super(host, service, ttl, new AnyChildShouldBeOk().withSkipDowntimes(), namespace, jcheckMark);
    }

    /**
     * Создать juggler-проверку для дочерних событий с одинаковым именем, но различными хостами источниками
     *
     * @param targetHost    результирующий хост
     * @param targetService имя результирующего события
     * @param children      дочерние события
     * @param ttl           TTL проверки
     * @param namespace     имя пространста имен проверки для ограничения доступа
     * @param jcheckMark    метка проверки
     * @throws NullPointerException
     * @throws IllegalArgumentException
     */
    public SingleEventFromAnyHostCheck(String targetHost, String targetService,
                                       Collection<JugglerChild> children, Duration ttl,
                                       String namespace, String jcheckMark) {
        this(targetHost, targetService, ttl, namespace, jcheckMark);
        this.withChildren(children);
    }

    /**
     * Создать juggler-проверку для дочерних событий с одинаковым именем, но различными хостами источниками
     *
     * @param targetHost           результирующий хост
     * @param targetService        имя результирующего события
     * @param sourceHostsOrCGroups список хостов или кондукторных групп источника событий
     * @param sourceService        имя исходного события
     * @param ttl                  TTL проверки
     * @param namespace            имя пространста имен проверки для ограничения доступа
     * @param jcheckMark           метка проверки
     * @throws NullPointerException
     * @throws IllegalArgumentException
     */
    public SingleEventFromAnyHostCheck(String targetHost, String targetService, List<String> sourceHostsOrCGroups,
                                       String sourceService, Duration ttl, String namespace, String jcheckMark) {
        this(targetHost, targetService, ttl, namespace, jcheckMark);
        checkChildrenHost(sourceHostsOrCGroups);
        checkName(sourceService);
        sourceHostsOrCGroups.forEach(host -> this.withChild(new JugglerChild(host, sourceService)));
    }

    /**
     * Создать juggler-проверку для дочерних событий с одинаковым именем, но различными хостами источниками,
     * имя результирующей проверки совпадает с дочерними
     *
     * @param targetHost           результирующий хост
     * @param service              имя дочернего и результирующего событий
     * @param sourceHostsOrCGroups список хостов или кондукторных групп источника событий
     * @param ttl                  TTL проверки
     * @param namespace            имя пространста имен проверки для ограничения доступа
     * @param jcheckMark           метка проверки
     * @throws NullPointerException
     * @throws IllegalArgumentException
     */
    public SingleEventFromAnyHostCheck(String targetHost, String service, List<String> sourceHostsOrCGroups,
                                       Duration ttl, String namespace, String jcheckMark) {
        this(targetHost, service, sourceHostsOrCGroups, service, ttl, namespace, jcheckMark);
    }

    /**
     * Создать juggler-проверку, где исходный хост (или кондукторная группа) - одна,
     * а имена исходного и результирующего событий - совпадает
     *
     * @param targetHost         результирующий хост
     * @param service            имя сервиса
     * @param sourceHostOrCGroup исходный хост или кондукторная группа
     * @param ttl                TTL проверки
     * @param namespace          имя пространста имен проверки для ограничения доступа
     * @param jcheckMark         метка проверки
     * @throws NullPointerException
     * @throws IllegalArgumentException
     */
    public SingleEventFromAnyHostCheck(String targetHost, String service, String sourceHostOrCGroup, Duration ttl,
                                       String namespace, String jcheckMark) {
        this(targetHost, service, Collections.singletonList(sourceHostOrCGroup), service, ttl, namespace, jcheckMark);
    }
}
