package ru.yandex.direct.ansiblejuggler.model.notifications;

import java.util.Arrays;
import java.util.Collection;
import java.util.LinkedHashSet;
import java.util.Objects;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.annotation.JsonPropertyOrder;

import ru.yandex.direct.juggler.JugglerStatus;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.base.Preconditions.checkState;

/**
 * Уведомление при смене статуса или описания проверки.
 * <p>
 * Не все параметры не поддержаны в оберетке.
 * Частично поддержан {@code status} - без возможности указания переходов между статусами.
 *
 * @see <a href=https://nda.ya.ru/3UZBgp>Опции для шаблонов on_status_change и on_desc_change</a>
 */
@ParametersAreNonnullByDefault
@JsonPropertyOrder({"template", "login", "status", "method"})
public class OnChangeNotification extends JugglerNotification {

    @JsonProperty("login")
    private Set<String> recipients;

    @JsonProperty("status")
    private Set<JugglerStatus> statuses;

    @JsonProperty("method")
    private Set<NotificationMethod> methods;

    public OnChangeNotification(ChangeType type) {
        super(type.getModule());
        recipients = new LinkedHashSet<>();
        statuses = new LinkedHashSet<>();
        methods = new LinkedHashSet<>();
    }

    /**
     * Добавить к нотификации метод ее доставки
     *
     * @param method метод доставки
     * @return текущая нотификация
     * @throws NullPointerException  если указанный метод равен {@code null}
     * @throws IllegalStateException если указанный метод уже содержится в нотификации
     */
    public OnChangeNotification withNotificationMethod(NotificationMethod method) {
        checkNotNull(method, "method cannot be null");
        checkState(!this.methods.contains(method), "notification already contains this method");
        this.methods.add(method);
        return this;
    }

    /**
     * Добавить к нотификации методы ее доставки
     *
     * @param methods метод доставки
     * @return текущая нотификация
     * @see #withNotificationMethod(NotificationMethod)
     */
    public OnChangeNotification withNotificationMethods(NotificationMethod... methods) {
        checkNotNull(methods, "methods cannot be null");
        Arrays.stream(methods).forEach(this::withNotificationMethod);
        return this;
    }

    /**
     * Добавить к нотификации получателя
     *
     * @param recipient логин получателя уведомления или имя телеграмной группы
     * @return текущая нотификация
     * @throws NullPointerException     если указанный получатель равен {@code null}
     * @throws IllegalArgumentException если получатель пустой
     * @throws IllegalStateException    если получатель уже содержится в нотификации
     */
    public OnChangeNotification withRecipient(String recipient) {
        checkNotNull(recipient, "recipient cannot be null");
        checkArgument(!recipient.isEmpty(), "recipient cannot be empty");
        checkState(!this.recipients.contains(recipient), "notification already contains this recipient");
        this.recipients.add(recipient);
        return this;
    }

    /**
     * Добавить к нотификации получателей
     *
     * @param recipients логины (или имена телеграммных групп) получателей уведомления
     * @return текущая нотификация
     * @see #withRecipient(String)
     */
    public OnChangeNotification withRecipients(Collection<String> recipients) {
        checkNotNull(recipients, "recipients cannot be null");
        recipients.forEach(this::withRecipient);
        return this;
    }

    /**
     * Добавить к нотификации ожидаемый статус проверки
     *
     * @param status статус проверки, для которых будет приходить уведомление
     * @return текущая нотификация
     * @throws NullPointerException  если указанный статус равен {@code null}
     * @throws IllegalStateException если указанный статус уже содержится в нотификации
     */
    public OnChangeNotification withStatus(JugglerStatus status) {
        checkNotNull(status, "status cannot be null");
        checkState(!this.statuses.contains(status), "notification already contains this status");
        this.statuses.add(status);
        return this;
    }

    /**
     * Добавить к нотификации ожидаемые статусы проверки
     *
     * @param statuses статусы проверки, для которых будет приходить уведомление
     * @return текущая нотификация
     * @see #withStatus(JugglerStatus)
     */
    public OnChangeNotification withStatuses(JugglerStatus... statuses) {
        checkNotNull(statuses, "statuses cannot be null");
        Arrays.stream(statuses).forEach(this::withStatus);
        return this;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        OnChangeNotification that = (OnChangeNotification) o;
        return Objects.equals(recipients, that.recipients) &&
                Objects.equals(statuses, that.statuses) &&
                Objects.equals(methods, that.methods);
    }

    @Override
    public int hashCode() {
        return Objects.hash(recipients, statuses, methods);
    }
}
