package ru.yandex.direct.appmetrika;

import java.util.List;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.appmetrika.model.AppMetrikaClientException;
import ru.yandex.direct.appmetrika.model.Platform;
import ru.yandex.direct.appmetrika.model.request.UniversalCampaignRequest;
import ru.yandex.direct.appmetrika.model.response.Application;
import ru.yandex.direct.appmetrika.model.response.ApplicationsResponse;
import ru.yandex.direct.appmetrika.model.response.ClientEventsResponse;
import ru.yandex.direct.appmetrika.model.response.TypedEventInfo;
import ru.yandex.direct.appmetrika.model.response.TypedEventsResponse;
import ru.yandex.direct.appmetrika.model.response.UniversalCampaignInfo;
import ru.yandex.direct.appmetrika.model.response.UniversalCampaignResponse;
import ru.yandex.direct.asynchttp.ParallelFetcherFactory;
import ru.yandex.direct.asynchttp.Result;
import ru.yandex.direct.http.smart.annotations.Json;
import ru.yandex.direct.http.smart.core.Call;
import ru.yandex.direct.http.smart.core.Smart;
import ru.yandex.direct.http.smart.http.Body;
import ru.yandex.direct.http.smart.http.GET;
import ru.yandex.direct.http.smart.http.Headers;
import ru.yandex.direct.http.smart.http.POST;
import ru.yandex.direct.http.smart.http.Query;
import ru.yandex.direct.tvm.TvmIntegration;
import ru.yandex.direct.tvm.TvmService;

import static com.google.common.base.Preconditions.checkState;
import static ru.yandex.direct.http.smart.error.ErrorUtils.checkResultForErrors;

/**
 * Клиент для аппметрики
 * Примеры вызова аппметрики https://st.yandex-team.ru/DIRECT-136068#60268fdc140d331ea1f00060
 * Примеры запросов из гридов https://st.yandex-team.ru/DIRECT-146965#60bbb710719ca10f8ae612b5
 */
@ParametersAreNonnullByDefault
public class AppMetrikaClient {
    private final Api api;

    public AppMetrikaClient(String url, TvmIntegration tvmIntegration, TvmService tvmService,
                            ParallelFetcherFactory parallelFetcherFactory) {
        api = createApi(url, tvmIntegration, tvmService, parallelFetcherFactory);
    }

    private Api createApi(String url, TvmIntegration tvmIntegration, TvmService tvmService,
                          ParallelFetcherFactory parallelFetcherFactory) {
        return Smart.builder()
                .withParallelFetcherFactory(parallelFetcherFactory)
                .withProfileName("appmetrika_client")
                .useTvm(tvmIntegration, tvmService)
                .withBaseUrl(url)
                .build()
                .create(Api.class);
    }

    public List<Application> getApplications(long uid, @Nullable String bundleId, @Nullable Platform platform,
                                             @Nullable String mask, @Nullable Integer limit, @Nullable Integer offset) {
        checkState((bundleId == null) == (platform == null), "bundleId and platform should be sent together");
        Result<ApplicationsResponse> result =
                api.getApplications(uid, bundleId, platform, mask, limit, offset).execute();
        checkResultForErrors(result, AppMetrikaClientException::new);
        return result.getSuccess().getApplications();
    }

    public List<String> getClientEvents(long operatorUid, long appId) {
        return getClientEvents(operatorUid, appId, null, null, null);
    }

    public List<String> getClientEvents(long operatorUid, long appId, @Nullable String mask, @Nullable Integer limit,
                                        @Nullable Integer offset) {
        Result<ClientEventsResponse> result = api.getClientEvents(operatorUid, appId, mask, limit, offset).execute();
        checkResultForErrors(result, AppMetrikaClientException::new);
        return result.getSuccess().getClientEventsInfo().getEvents();
    }

    public List<TypedEventInfo> getTypedEvents(long operatorUid, long appId) {
        Result<TypedEventsResponse> result = api.getTypedEvents(operatorUid, appId).execute();
        checkResultForErrors(result, AppMetrikaClientException::new);
        return result.getSuccess().getEvents();
    }

    public UniversalCampaignInfo getUniversalCampaign(long operatorUid, long appId,
                                                      UniversalCampaignRequest universalCampaignRequest) {
        Result<UniversalCampaignResponse> result = api.getUniversalCampaign(operatorUid, appId,
                universalCampaignRequest).execute();
        checkResultForErrors(result, AppMetrikaClientException::new);
        return result.getSuccess().getCampaign();
    }

    public interface Api {
        @GET("/applications")
        @Json
        Call<ApplicationsResponse> getApplications(@Query("uid") long uid,
                                                   @Nullable @Query("bundle_id") String bundleId,
                                                   @Nullable @Query("platform") Platform platform,
                                                   @Nullable @Query("mask") String mask,
                                                   @Nullable @Query("limit") Integer limit,
                                                   @Nullable @Query("offset") Integer offset);

        @GET("/client_events")
        @Json
        Call<ClientEventsResponse> getClientEvents(@Query("uid") long uid, @Query("app_id") long appId,
                                                   @Nullable @Query("mask") String mask,
                                                   @Nullable @Query("limit") Integer limit,
                                                   @Nullable @Query("offset") Integer offset);

        @GET("/typed_events")
        @Json
        Call<TypedEventsResponse> getTypedEvents(@Query("uid") long uid, @Query("app_id") long appId);

        @POST("/universal_campaign")
        @Json
        @Headers("Content-Type: application/json")
        Call<UniversalCampaignResponse> getUniversalCampaign(@Query("uid") long uid,
                                                             @Query("app_id") long appId,
                                                             @Body @Json UniversalCampaignRequest universalCampaignRequest);
    }
}
