package ru.yandex.direct.asynchttp;

import java.time.Duration;

import ru.yandex.monlib.metrics.registry.MetricRegistry;

import static com.google.common.base.Preconditions.checkArgument;

@SuppressWarnings("WeakerAccess")
public class FetcherSettings {
    public static final Duration DEFAULT_CONNECTION_TIMEOUT = Duration.ofMillis(100);
    public static final Duration DEFAULT_REQUEST_TIMEOUT = Duration.ofMinutes(5);
    public static final double NO_TOTAL_RETRIES_LIMIT = 0.0;

    private int parallel = 1;
    private Duration globalTimeout;
    private Duration softTimeout;
    private Duration requestTimeout = DEFAULT_REQUEST_TIMEOUT;
    private Duration connectTimeout = DEFAULT_CONNECTION_TIMEOUT;
    private double totalRetriesCoef = NO_TOTAL_RETRIES_LIMIT;
    private int requestRetries = 0;

    private boolean failFast = false;
    private MetricRegistry metricRegistry;

    public FetcherSettings() {
        // Не нужно терять возможность создать экземпляр без параметра
    }

    /**
     * Важно: конструктор копирования не копирует onResponseCallback.
     */
    public FetcherSettings(FetcherSettings baseSettings) {
        this.parallel = baseSettings.parallel;
        this.globalTimeout = baseSettings.globalTimeout;
        this.softTimeout = baseSettings.softTimeout;
        this.requestTimeout = baseSettings.requestTimeout;
        this.connectTimeout = baseSettings.connectTimeout;
        this.totalRetriesCoef = baseSettings.totalRetriesCoef;
        this.requestRetries = baseSettings.requestRetries;
        this.failFast = baseSettings.failFast;
        this.metricRegistry = baseSettings.metricRegistry;
    }

    public int getParallel() {
        return parallel;
    }

    /**
     * Задает число параллельно выполняемых запросов в рамках одного инстанса {@link ParallelFetcher}.
     *
     * @param parallel число параллельно выполняемых запросов
     */
    @SuppressWarnings("SameParameterValue")
    public FetcherSettings withParallel(int parallel) {
        checkArgument(parallel >= 1, "Parallel must be greater or equal than 1");
        this.parallel = parallel;
        return this;
    }

    public Duration getGlobalTimeout() {
        return globalTimeout;
    }

    public FetcherSettings withGlobalTimeout(Duration globalTimeout) {
        this.globalTimeout = globalTimeout;
        return this;
    }

    public Duration getSoftTimeout() {
        return softTimeout;
    }

    /**
     * Задает "мягкий" таймаут для запроса. По soft-timeout сервис посылает еще один запрос не прерывая предыдущий,
     * в надежде попасть на более быструю ноду.
     * Для перезапросов (requestRetries) soft-timeout не отсчитывается.
     * Для использования soft-timeout значение parallel должно быть больше 1, так как новый запрос создает еще один
     * поток.
     * <p>
     * Типичный сценарий использования - одновременно уходят N запросов на сервис, у которого 99% запросов обработаются
     * за 1 сек, а 1% - за 100 сек, независимо от запроса.
     *
     * @param softTimeout таймаут
     */
    public FetcherSettings withSoftTimeout(Duration softTimeout) {
        this.softTimeout = softTimeout;
        return this;
    }

    public Duration getRequestTimeout() {
        return requestTimeout;
    }

    public FetcherSettings withRequestTimeout(Duration requestTimeout) {
        this.requestTimeout = requestTimeout;
        return this;
    }

    public Duration getConnectTimeout() {
        return connectTimeout;
    }

    public FetcherSettings withConnectTimeout(Duration connectionTimeout) {
        this.connectTimeout = connectionTimeout;
        return this;
    }

    public double getTotalRetriesCoef() {
        return totalRetriesCoef;
    }

    public FetcherSettings withTotalRetriesCoef(double totalRetriesCoef) {
        this.totalRetriesCoef = totalRetriesCoef;
        return this;
    }

    public int getRequestRetries() {
        return requestRetries;
    }

    /**
     * Задает количество retry для запроса. Первый запрос retry не считается, т.е. будет выполнено максимум
     * {@code (requestRetries + 1)} запросов
     *
     * @param requestRetries количество retry для запроса
     */
    @SuppressWarnings("SameParameterValue")
    public FetcherSettings withRequestRetries(int requestRetries) {
        this.requestRetries = requestRetries;
        return this;
    }

    public boolean isFailFast() {
        return failFast;
    }

    public FetcherSettings withFailFast(boolean failFast) {
        this.failFast = failFast;
        return this;
    }

    public MetricRegistry getMetricRegistry() {
        return metricRegistry;
    }

    public FetcherSettings withMetricRegistry(MetricRegistry metricRegistry) {
        this.metricRegistry = metricRegistry;
        return this;
    }
}
