package ru.yandex.direct.asynchttp;

import java.time.Duration;
import java.util.ArrayDeque;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Queue;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Executor;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.primitives.Ints;
import one.util.streamex.StreamEx;
import org.asynchttpclient.AsyncHttpClient;
import org.asynchttpclient.BoundRequestBuilder;
import org.asynchttpclient.ListenableFuture;
import org.asynchttpclient.Param;
import org.asynchttpclient.Request;
import org.asynchttpclient.Response;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import static com.google.common.base.Preconditions.checkState;
import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.asynchttp.FetcherSettings.NO_TOTAL_RETRIES_LIMIT;

/**
 * !!! При изменениях надо запускать тесты FetcherTest вручную, в ci они не запускаются, т. к. слишком чувствительны
 * !!! к нехватке процессорного времени.
 *
 * Система для параллельных HTTP-запросов с настраиваемыми таймаутами
 *
 * @param <T> — класс запрашиваемых данных
 */
@SuppressWarnings("WeakerAccess")
@ParametersAreNonnullByDefault
public class ParallelFetcher<T> implements AutoCloseable {
    private static final Executor IMMEDIATE_EXECUTOR = Runnable::run;
    private static final Logger logger = LoggerFactory.getLogger(ParallelFetcher.class);
    private static final long MAX_WAIT_TIMEOUT_MILLIS = Duration.ofSeconds(1).toMillis();
    public static final int NO_GLOBAL_RETRY_LIMIT = Integer.MAX_VALUE;

    private final FetcherSettings settings;

    private final Queue<RequestData<T>> todo = new ArrayDeque<>();
    private final BlockingQueue<RequestWithResponse<T>> resultQueue = new LinkedBlockingQueue<>();
    private final Queue<RequestData<T>> softRetryQueue = new ArrayDeque<>();
    private final Map<Long, Result<T>> results = new HashMap<>();
    private final AsyncHttpClient asyncHttpClient;
    private int globalRetries = 0;
    private int globalRetriesLimit = NO_GLOBAL_RETRY_LIMIT;
    private long borderTimeNanos;
    private int callsInFly;
    private boolean singleRequestFailPresent = false;
    private ParallelFetcherMetrics metrics;

    /**
     * Конструктор {@link ParallelFetcher}, использующий инициализированный экземпляр {@link AsyncHttpClient}.
     *
     * <b>Внимание:</b> параметры {@link FetcherSettings#getConnectTimeout()} и
     * {@link FetcherSettings#getRequestTimeout()} из {@code settings}при этом не используются
     *
     * @param settings        {@link FetcherSettings} с настройками
     * @param asyncHttpClient {@link AsyncHttpClient}
     */
    public ParallelFetcher(FetcherSettings settings, AsyncHttpClient asyncHttpClient) {
        this.settings = Objects.requireNonNull(settings);
        this.asyncHttpClient = Objects.requireNonNull(asyncHttpClient);
        this.metrics = new ParallelFetcherMetrics(settings.getMetricRegistry());
    }

    /**
     * Выпоолнение одиночного запроса с перехватом внутренних исключений в случае ошибок и заменой общей ошибкой.
     *
     * @param request - запрос
     * @return результат выполнения запроса
     */
    public Result<T> executeWithErrorsProcessing(ParsableRequest<T> request) {
        Result<T> result;

        try {
            result = execute(request);
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new AsyncHttpExecuteException("http execute server exception", e);
        }

        if (result.getSuccess() == null) {
            throw new AsyncHttpExecuteException("http execute server error", result.getErrors());
        }

        return result;
    }

    public Result<T> execute(ParsableRequest<T> request) throws InterruptedException {
        return execute(Collections.singletonList(request)).get(request.getId());
    }

    public Map<Long, Result<T>> execute(List<? extends ParsableRequest<T>> requests)
            throws InterruptedException {
        List<RequestData<T>> reqsData = requests.stream().map(RequestData::new).collect(toList());
        todo.addAll(reqsData);

        borderTimeNanos = settings.getGlobalTimeout() != null
                ? System.nanoTime() + settings.getGlobalTimeout().toNanos()
                : Long.MAX_VALUE;
        globalRetriesLimit = getGlobalRetriesLimit(requests.size());
        callsInFly = 0;
        boolean failFast = settings.isFailFast();

        RequestWithResponse<T> peekResult = null;
        while (true) {
            if (peekResult != null) {
                // значит, на предыдущей итерации цикла дождались результата
                callsInFly--;
                processResult(peekResult);
            }
            logger.debug("[main loop] resultQueue.size(): {}, todo.size(): {}, callsInFly: {}",
                    resultQueue.size(), todo.size(), callsInFly);
            // обрабатываем результаты
            while (!resultQueue.isEmpty()) {
                RequestWithResponse<T> result = resultQueue.poll();
                callsInFly--;
                processResult(result);
            }
            logger.debug("[main loop][resultQueue processed] todo.size(): {}", todo.size());

            // если вся работа сделана - выходим
            if (todo.isEmpty() && callsInFly == 0) {
                break;
            }

            // проверка на глобальный таймаут
            if (borderTimeNanos < System.nanoTime()) {
                logger.debug("Timeout occurred for batch request. Terminate all running requests");
                reqsData.forEach(RequestData::abort);
                TimeoutException error = new TimeoutException(
                        String.format("Timeout %dms occurred", settings.getGlobalTimeout().toMillis()));
                fillIncompleteResultsWithError(reqsData, error);
                break;
            }

            if (failFast && singleRequestFailPresent) {
                logger.info("There are at least one failed request and failFast is set to true. "
                        + "Abort requests processing");
                reqsData.forEach(RequestData::abort);
                AsyncHttpExecutionAbortedException error = new AsyncHttpExecutionAbortedException();
                fillIncompleteResultsWithError(reqsData, error);
                break;
            }

            // если есть слоты - добавляем запросы из todoQueue и softRetryQueue
            while (callsInFly < settings.getParallel()) {
                if (!todo.isEmpty()) {
                    RequestData<T> requestData = todo.poll();
                    if (results.containsKey(requestData.getRequest().getId())
                            && results.get(requestData.getRequest().getId()).getSuccess() != null) {
                        // этот запрос уже обработали, пропускаем
                        continue;
                    }
                    sendRequest(asyncHttpClient, requestData);
                    metrics.incRequests();

                    // регистрируем softTimeout для текущего запроса, если он ещё не был установлен
                    if (settings.getSoftTimeout() != null && requestData.getSoftRetryTimeNanos() == 0) {
                        requestData.setSoftRetryTimeNanos(System.nanoTime() + settings.getSoftTimeout().toNanos());
                        softRetryQueue.add(requestData);
                    }
                    callsInFly++;
                } else if (!softRetryQueue.isEmpty()
                        && softRetryQueue.peek().getSoftRetryTimeNanos() <= System.nanoTime()) {
                    // добавляем в todoQueue запросы по soft-timeout, на следующей итерации цикла они поступят в работу
                    RequestData<T> requestData = softRetryQueue.poll();
                    // делаем retry если первый запрос всё ещё выполняется, иначе количество retries > 0
                    if (requestData.getRetries() == 0) {
                        retryRequest(requestData, true);
                    }
                } else {
                    break;
                }
            }

            // Блокируемся либо до прихода результата, либо до времени срабатывания soft/global timeout'а
            peekResult = resultQueue.poll(timeTillNextTimeout(), TimeUnit.MILLISECONDS);
        }

        logger.debug("batch done");
        return results;
    }

    private void fillIncompleteResultsWithError(List<RequestData<T>> reqsData, Exception error) {
        // добавим в results данные по запросам, которые не успели обработать
        reqsData.stream()
                .map(RequestData::getRequest)
                .map(ParsableRequest::getId)
                .distinct()
                .forEach(id -> {
                    // добавляем ошибку только если результатов по данному запросу нет
                    // иначе он уже содержит или результат успешного выполнения, или ошибку
                    if (!results.containsKey(id)) {
                        Result<T> result = new Result<>(id);
                        result.addError(error);
                        results.put(id, result);
                    }
                });
    }

    /**
     * @param requestCount количество запросов в вызове
     * @return лимит retry на вызов. Вычисляется как {@code floor(requestRetries + retryBudgetPercent * requestCount)}
     * Формула взята из Perl'а. Смысл такой, что на большом количестве запросов
     * примерно получаем {@code retryBudgetPercent * requestCount}, а на маленьком {@code requestRetries}
     */
    private int getGlobalRetriesLimit(int requestCount) {
        int requestRetries = settings.getRequestRetries();

        Double totalRetriesCoef = settings.getTotalRetriesCoef();
        if (totalRetriesCoef == NO_TOTAL_RETRIES_LIMIT) {
            return NO_GLOBAL_RETRY_LIMIT;
        }

        // приведение double к int в случае переполнения даст Integer.MAX_VALUE
        return (int) Math.floor(requestRetries + totalRetriesCoef * requestCount);
    }

    private void sendRequest(AsyncHttpClient asyncHttpClient, RequestData<T> requestData) {
        logger.debug("send {}", requestData);

        Request ahcRequest = requestData.getRequest().getAHCRequest();
        if (logger.isTraceEnabled()) {
            // в RequestDate#toString() не попадает тело POST запроса. Выводим его при уровне trace тут
            String formParams;
            formParams = StreamEx.ofNullable(ahcRequest.getFormParams())
                    .flatMap(Collection::stream)
                    .mapToEntry(Param::getName, Param::getValue)
                    .grouping().toString();
            logger.trace("method: {}, url: {}, body: {}, params: {}",
                    ahcRequest.getMethod(), ahcRequest.getUrl(), ahcRequest.getStringData(), formParams);
        }
        BoundRequestBuilder reqBuilder = asyncHttpClient.prepareRequest(ahcRequest);
        // если в запросе нет таймаута - выставляем его из настроек
        if (ahcRequest.getRequestTimeout() == 0 && settings.getRequestTimeout() != null) {
            reqBuilder.setRequestTimeout(Ints.saturatedCast(settings.getRequestTimeout().toMillis()));
        }
        ListenableFuture<Response> call = reqBuilder.execute();
        long startTime = System.nanoTime();

        // сохраняем вызов, чтобы иметь возможность сделать abort()
        requestData.addCall(call);
        // регистрируем listener для удаления вызова после завершения
        call.addListener(() -> requestData.removeCall(call), IMMEDIATE_EXECUTOR);

        // добавляем вызов callback'а
        call.addListener(() -> {
            Response result = null;
            Exception exception = null;
            long responseTimeMillis = TimeUnit.NANOSECONDS.toMillis(System.nanoTime() - startTime);
            metrics.recordResponseTime(responseTimeMillis);
            logger.debug("response for request {}", requestData.getRequest().getId());
            try {
                result = call.get();
            } catch (InterruptedException | ExecutionException e) {
                logger.info("ExecutionError while processing request {}", requestData, e);
                exception = e;
            }
            RequestWithResponse<T> requestWithResponse = new RequestWithResponse<>(requestData, result, exception);

            resultQueue.add(requestWithResponse);
        }, IMMEDIATE_EXECUTOR);
    }

    private long timeTillNextTimeout() {
        long waitTimeoutMillis = MAX_WAIT_TIMEOUT_MILLIS;
        long timeoutToBorderTimeMillis = Duration.ofNanos(borderTimeNanos - System.nanoTime()).toMillis();
        waitTimeoutMillis = Math.min(waitTimeoutMillis, timeoutToBorderTimeMillis);
        long requestTimeoutMillis = settings.getRequestTimeout().toMillis();
        waitTimeoutMillis = Math.min(waitTimeoutMillis, requestTimeoutMillis);
        if (!softRetryQueue.isEmpty() && callsInFly < settings.getParallel()) {
            long timeToSoftTimeoutMillis =
                    Duration.ofNanos(softRetryQueue.peek().getSoftRetryTimeNanos() - System.nanoTime()).toMillis();
            waitTimeoutMillis = Math.min(waitTimeoutMillis, timeToSoftTimeoutMillis);
        }
        return waitTimeoutMillis;
    }

    private void processResult(RequestWithResponse<T> result) {
        logger.trace("process result {}", result);

        RequestData<T> requestData = result.getRequestData();
        ParsableRequest<T> request = requestData.getRequest();
        long id = request.getId();

        Result<T> parsedResult = results.computeIfAbsent(id, Result::new);

        Exception exception;
        if (result.isSuccessful()) {
            metrics.incSuccesses();
            try {
                // пробуем распарсить
                T parsedResponse = request.getParseFunction().apply(result.getResponse());
                parsedResult.setSuccess(parsedResponse);

                // если успех, то прерываем прочие вызовы этого запроса
                requestData.abort();
                return;
            } catch (RuntimeException parseException) {
                logger.error("Error during response parsing. Request: {}", request, parseException);
                exception = parseException;
            }
        } else {
            checkState(result.getException() != null,
                    "result.getException() must not return null for unsuccessful result");
            Exception innerException = result.getException();
            // Оборачиваем ошибку в специальный враппер, хранящий исходный ответ - в нем может быть дополнительная
            // информация, в частности в body может быть раскрыта причина ошибки
            exception = new ErrorResponseWrapperException("Error during request",
                    result.getResponse(), innerException);

            // не логируем запросы, которые были прерваны нами через RequestData#abort()
            boolean aborted = ExecutionException.class.isInstance(innerException) &&
                    RequestExecutionAbortedException.class.isInstance(innerException.getCause());
            if (!aborted) {
                metrics.incFailures();
                logger.warn("Error \"{}\" on executing request {}", innerException.getMessage(), requestData);
            } else {
                metrics.incAborted();
            }
        }

        parsedResult.addError(exception);
        retryRequest(requestData, false);
    }

    /**
     * Повторяем запрос, если
     * <ul>
     * <li>он ещё не был выполнен</li>
     * <li>есть бюджет retry'ев на запрос</li>
     * <li>есть бюджет retry'ев на пачку запросов</li>
     * </ul>
     *
     * @param requestData {@link RequestData}
     */
    private void retryRequest(RequestData<T> requestData, boolean isSoftRetry) {
        if (results.containsKey(requestData.getRequest().getId())
                && results.get(requestData.getRequest().getId()).getSuccess() != null) {
            // результат уже получен, например, запросом, отправленным после softTimeout'а
            // не делаем ничего
            return;
        }
        if (requestData.getRetries() < settings.getRequestRetries()
                && globalRetries < globalRetriesLimit) {
            logger.info("let's retry request {}", requestData);
            if (isSoftRetry) {
                metrics.incSoftRetries();
            } else {
                metrics.incRetriesOnFailure();
            }
            globalRetries++;
            requestData.incrementRetries();
            todo.add(requestData);
        } else {
            if (settings.isFailFast()) {
                boolean noRunningCalls = requestData.getRelatedCallsInFlyCount() == 0;
                ParsableRequest<T> request = requestData.getRequest();
                Result<T> relatedResult = results.get(request.getId());
                boolean successResult = relatedResult != null && relatedResult.getSuccess() != null;
                if (noRunningCalls && !successResult) {
                    logger.debug("Can't process request with failFast=true. Abort execution. Request: {}", requestData);
                    // взведём флаг о наличии неуспешного запроса,
                    // чтобы быстро завершить обработку при failFast = true
                    singleRequestFailPresent = true;
                }
            }

            // если это softRetry - не страшно, error должен писаться при обработке результата
            logger.info("tries number exhaust");
        }
    }

    @Override
    public void close() {
    }

    @VisibleForTesting
    int getGlobalRetries() {
        return globalRetries;
    }

    @VisibleForTesting
    ParallelFetcherMetrics getMetrics() {
        return metrics;
    }

    // для разбирательства с количеством ретраев в проде
    public int getConfiguredRequestRetries() {
        return settings.getRequestRetries();
    }
}
