package ru.yandex.direct.asynchttp;

import javax.annotation.Nullable;

import com.google.common.annotations.VisibleForTesting;

import ru.yandex.monlib.metrics.histogram.Histograms;
import ru.yandex.monlib.metrics.primitives.Histogram;
import ru.yandex.monlib.metrics.primitives.Rate;
import ru.yandex.monlib.metrics.registry.MetricRegistry;

class ParallelFetcherMetrics {
    private Rate requestsSensor;
    private Rate successesSensor;
    private Rate failuresSensor;
    private Rate abortedSensor;
    private Rate softRetriesSensor;
    private Rate retriesOnFailureSensor;
    private Histogram responseTimeSensor;

    public ParallelFetcherMetrics(@Nullable MetricRegistry metricRegistry) {
        if (metricRegistry != null) {
            this.requestsSensor = metricRegistry.rate("requests");
            this.successesSensor = metricRegistry.rate("successes");
            this.failuresSensor = metricRegistry.rate("failures");
            this.abortedSensor = metricRegistry.rate("aborted");
            this.softRetriesSensor = metricRegistry.rate("soft_retries");
            this.retriesOnFailureSensor = metricRegistry.rate("retries_on_failure");
            this.responseTimeSensor = metricRegistry.histogramRate(
                    "response_time_millis",
                    // (-INF, 100ms], (100ms, 200ms], (200ms, 400ms], (400ms, 800ms], (800ms, 1.6s], (1.6s, 3.2ms],
                    // (3.2s, 6.4s], (6.4s, 12.8s], (12.8s, 25.6s], (25.6s, 51.2s], (51.2s, 102.4s], (102.4s, +INF)
                    Histograms.exponential(12, 2, 100)
            );
        }
    }

    void incRequests() {
        if (requestsSensor != null) {
            requestsSensor.inc();
        }
    }

    void incSuccesses() {
        if (successesSensor != null) {
            successesSensor.inc();
        }
    }

    void incFailures() {
        if (failuresSensor != null) {
            failuresSensor.inc();
        }
    }

    void incAborted() {
        if (abortedSensor != null) {
            abortedSensor.inc();
        }
    }

    void incSoftRetries() {
        if (softRetriesSensor != null) {
            softRetriesSensor.inc();
        }
    }

    void incRetriesOnFailure() {
        if (retriesOnFailureSensor != null) {
            retriesOnFailureSensor.inc();
        }
    }

    void recordResponseTime(long responseTimeMillis) {
        if (responseTimeSensor != null) {
            responseTimeSensor.record(responseTimeMillis);
        }
    }

    @VisibleForTesting
    Long getRequests() {
        return requestsSensor != null ? requestsSensor.get() : null;
    }

    @VisibleForTesting
    Long getSuccesses() {
        return successesSensor != null ? successesSensor.get() : null;
    }

    @VisibleForTesting
    Long getFailures() {
        return failuresSensor != null ? failuresSensor.get() : null;
    }

    @VisibleForTesting
    Long getAborted() {
        return abortedSensor != null ? abortedSensor.get() : null;
    }

    @VisibleForTesting
    Long getSoftRetries() {
        return softRetriesSensor != null ? softRetriesSensor.get() : null;
    }

    @VisibleForTesting
    Long getRetriesOnFailure() {
        return retriesOnFailureSensor != null ? retriesOnFailureSensor.get() : null;
    }
}
