package ru.yandex.direct.asynchttp;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.asynchttpclient.Response;

import static com.google.common.base.Preconditions.checkArgument;

@ParametersAreNonnullByDefault
class RequestWithResponse<T> {
    private final RequestData<T> requestData;
    private final Response response;
    private final Exception exception;

    RequestWithResponse(RequestData<T> requestData, @Nullable Response response, @Nullable Exception exception) {
        this.requestData = requestData;
        this.response = response;
        this.exception = exception;
        checkArgument(exception != null || response != null,
                "Got both exception and response null from request call for request with id %s", requestData
                        .getRequest().getId());
    }

    @Nullable
    Response getResponse() {
        return response;
    }

    @Nullable
    Exception getException() {
        if (exception != null) {
            return exception;
        }
        if (isSuccessful()) {
            return null;
        }
        return new RuntimeException("Call was unsuccessful. Response: " + response);
    }


    RequestData<T> getRequestData() {
        return requestData;
    }

    boolean isSuccessful() {
        return exception == null && response != null &&
                requestData.getRequest().isParsableResponse(response);
    }

    @Override
    public String toString() {
        return "ReqResultPair{" +
                "requestData=" + requestData +
                ", exception=" + (exception == null ? null : exception.getLocalizedMessage()) +
                ", response=" + printedResponse(response) +
                '}';
    }

    /**
     * Наглядное представление ответа с телом
     */
    private static StringBuffer printedResponse(Response response) {
        StringBuffer res = new StringBuffer();
        res.append("{code: ").append(response.getStatusCode()).append(", body:");
        if (response.hasResponseBody()) {
            res.append('\n').append(response.getResponseBody()).append("\n}");
        } else {
            res.append(" null }");
        }
        return res;
    }
}
