package ru.yandex.direct.asynchttp;

import java.util.Optional;
import java.util.function.Function;
import java.util.stream.Collectors;

import org.apache.commons.lang3.builder.ToStringBuilder;
import org.apache.commons.lang3.builder.ToStringStyle;

import ru.yandex.direct.utils.JsonUtils;

public class ResultUtils {
    public static <T, E extends Exception> void checkResultForErrors(Result<T> result,
                                                                     Function<String, E> exceptionProvider,
                                                                     Object... args) throws E {
        if (result.getErrors() != null) {
            var exception = exceptionProvider.apply(ResultUtils.getErrorMessage(result, args));
            result.getErrors().forEach(exception::addSuppressed);
            throw exception;
        }
    }

    /**
     * Сериализовать ошибки из {@link Result#getErrors()} в строку пригодную для лога
     *
     * @param args дополнительные данные для логирования
     */
    public static <T> String getErrorMessage(Result<T> result, Object... args) {
        var errorsString = "Got error while http call: " +
                result.getErrors().stream()
                        .map(e -> e.getCause() + (e.getMessage() != null ? " (" + e.getMessage() + ")" : ""))
                        .collect(Collectors.joining("; "));
        return Optional.ofNullable(args)
                .filter(a -> a.length > 0)
                .map(ResultUtils::toString)
                .map(s -> errorsString + "; Call arguments: " + s)
                .orElse(errorsString);
    }

    private static String toString(Object arg) {
        try {
            return JsonUtils.toJson(arg);
        } catch (IllegalArgumentException e) {
            return ToStringBuilder.reflectionToString(arg, ToStringStyle.JSON_STYLE, true);
        }
    }

}
