package ru.yandex.direct.avatars.client;

import java.net.URISyntaxException;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.node.ObjectNode;
import org.apache.http.client.utils.URIBuilder;
import org.asynchttpclient.request.body.multipart.ByteArrayPart;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.asynchttp.ParallelFetcherFactory;
import ru.yandex.direct.asynchttp.Result;
import ru.yandex.direct.avatars.client.exception.AvatarsClientCommonException;
import ru.yandex.direct.avatars.client.model.AvatarId;
import ru.yandex.direct.avatars.client.model.AvatarInfo;
import ru.yandex.direct.avatars.client.model.answer.StatusCode;
import ru.yandex.direct.avatars.client.model.answer.UploadImageResponse;
import ru.yandex.direct.avatars.config.AvatarsConfig;
import ru.yandex.direct.avatars.config.ServerConfig;
import ru.yandex.direct.http.smart.converter.ResponseConverterFactory;
import ru.yandex.direct.http.smart.core.Call;
import ru.yandex.direct.http.smart.core.Smart;
import ru.yandex.direct.tvm.TvmIntegration;
import ru.yandex.direct.tvm.TvmService;
import ru.yandex.direct.utils.JsonUtils;

import static ru.yandex.direct.http.smart.error.ErrorUtils.checkResultForErrors;

@ParametersAreNonnullByDefault
public class AvatarsClient {
    private static final Logger logger = LoggerFactory.getLogger(AvatarsClient.class);
    private static final String OPERATION_SEPARATOR = "-";
    private static final String URI_PATH_SEPARATOR = "/";
    private static final String READ_OPERATION = "get";
    private final AvatarsConfig conf;
    private final ServerConfig readServerConfig;
    private final Api api;
    private final String uploadNamespace;
    private final Integer priorityPassParam; // https://wiki.yandex-team.ru/mds/avatars/#obrabotkaprioritetnyxzaprosov

    public AvatarsClient(AvatarsConfig conf, ParallelFetcherFactory parallelFetcherFactory,
                         @Nullable TvmIntegration tvmIntegration, @Nullable TvmService tvmService) {
        this.conf = conf;
        readServerConfig = conf.getReadServerConfig();
        api = createApi(parallelFetcherFactory, tvmIntegration, tvmService);
        uploadNamespace = conf.getUploadNamespace();
        priorityPassParam = conf.getPriorityPass() ? 1 : null;
    }

    private Api createApi(ParallelFetcherFactory parallelFetcherFactory,
                          @Nullable TvmIntegration tvmIntegration,
                          @Nullable TvmService tvmService) {
        ServerConfig writeConf = conf.getWriteServerConfig();
        String url = writeConf.getServerSchema() + "://" + writeConf.getServerHost() + ":" + writeConf.getServerPort();
        Smart.Builder builder = Smart.builder();
        if (tvmIntegration != null && tvmService != null) {
            builder.useTvm(tvmIntegration, tvmService);
        }
        return builder
                .withParallelFetcherFactory(parallelFetcherFactory)
                .withResponseConverterFactory(ResponseConverterFactory.builder()
                        .addConverters(
                                new DeleteAvatarsResponseConverter(),
                                new UploadImageResponseConverter())
                        .build())
                .withProfileName("avatars_client")
                .withBaseUrl(url)
                .build()
                .create(Api.class);
    }


    /**
     * Загружает изображение на сервер.
     * Описание используемого в аватарнице API: https://wiki.yandex-team.ru/mds/avatars/#put
     *
     * @param key       ключ по которому загружается изображение.
     * @param imageBody тело картинки.
     * @return информация о загруженном изображении.
     */
    public AvatarInfo upload(String key, byte[] imageBody) {
        Result<UploadImageResponse> result =
                api.upload(uploadNamespace, new ByteArrayPart("file", imageBody), key, priorityPassParam).execute();
        checkResultForErrors(result, AvatarsClientCommonException::new);
        UploadImageResponse avatarResponse = result.getSuccess();
        String avatarMeta = filterAvatarMeta(avatarResponse.getMeta());
        return new AvatarInfo(conf.getUploadNamespace(), avatarResponse.getGroupId(), avatarResponse.getImageName(),
                avatarMeta, avatarResponse.getSizes());
    }

    /**
     * Загружает аватарку на сервер.
     * Описание используемого в аватарнице API: https://wiki.yandex-team.ru/mds/avatars/#put
     *
     * @param imageBody тело картинки.
     * @return идентификатор загруженной автарки.
     */
    public AvatarId upload(byte[] imageBody) {
        Result<UploadImageResponse> result =
                api.upload(uploadNamespace, new ByteArrayPart("file", imageBody), priorityPassParam).execute();
        checkResultForErrors(result, AvatarsClientCommonException::new);
        UploadImageResponse avatarResponse = result.getSuccess();
        return new AvatarId(conf.getUploadNamespace(), avatarResponse.getGroupId(), avatarResponse.getImageName());

    }

    /**
     * Загружает изображение по его урлу на сервер.
     * Описание используемого в аватарнице API: https://wiki.yandex-team.ru/mds/avatars/#put
     *
     * @param key      ключ по которому загружается изображение.
     * @param imageUrl url картинки.
     * @return информация о загруженном изображении.
     */
    public AvatarInfo uploadByUrl(String key, String imageUrl) {
        Result<UploadImageResponse> result =
                api.uploadByUrl(uploadNamespace, key, imageUrl, priorityPassParam).execute();
        checkResultForErrors(result, AvatarsClientCommonException::new);
        UploadImageResponse avatarResponse = result.getSuccess();
        String avatarMeta = filterAvatarMeta(avatarResponse.getMeta());
        return new AvatarInfo(conf.getUploadNamespace(), avatarResponse.getGroupId(), avatarResponse.getImageName(),
                avatarMeta, avatarResponse.getSizes());
    }

    /**
     * Удаляет с серверов заданную аватарку. Из-за особенностей работы МДС есть некоторые особенности в работе метода.
     * Описание используемого в аватарнице API: https://wiki.yandex-team.ru/mds/avatars/#delete
     *
     * @param avatarId идентификатор аватарки в МДС.
     * @return true - аватарка точно удалена со всех серверов, всё, можно про неё забыть.
     * false - аватарка удалена с серверов МДС, но, возможно (!) не со всех шардов. Нужно пробовать удалить её ещё
     * раз (через час или через день) пока функция не вернёт ответ true.
     */
    public boolean delete(AvatarId avatarId) {
        Call<StatusCode> delete = api.delete(uploadNamespace, avatarId.getGroupId(), avatarId.getKey());
        Result<StatusCode> result = delete.execute();
        checkResultForErrors(result, AvatarsClientCommonException::new);
        int statusCode = result.getSuccess().getValue();
        switch (statusCode) {
            case 200:
            case 404:
                return true;
            case 202:
                return false;
            default:
                throw new AvatarsClientCommonException(
                        "Server returned unexpected code: " + statusCode + ".");
        }
    }

    /**
     * По externalId ("$namespace/$group-id/$key") аватарки создаёт URL на чтение аватарки.
     *
     * @param avatarExternalId Идентификатор аватарки в аватарнице МДС.
     * @return URL на чтение аватарки.
     */
    public String getReadUrl(String avatarExternalId, String avatarSizeConfigName) {
        String uploadPath = URI_PATH_SEPARATOR + READ_OPERATION + OPERATION_SEPARATOR
                + avatarExternalId + URI_PATH_SEPARATOR + avatarSizeConfigName;
        return getReadUrl(uploadPath);
    }

    public String getReadUrl(String avatarsPath) {
        try {
            return new URIBuilder()
                    .setScheme(readServerConfig.getServerSchema())
                    .setHost(readServerConfig.getServerHost())
                    .setPort(readServerConfig.getServerPort())
                    .setPath(avatarsPath)
                    .build()
                    .toASCIIString();
        } catch (URISyntaxException e) {
            logger.error("Request was stopped with URISyntaxException: {}", e.getMessage());
            throw new AvatarsClientCommonException(e);
        }
    }

    public AvatarsConfig getConf() {
        return conf;
    }

    /**
     * Фильтрует поля с метаданными аватарницы нейронки (NNetFeatures) и классификаторы (NeuralNetClasses)
     * из ответа аватарницы, так как они требуют большое количество места для хранения
     */
    private static String filterAvatarMeta(String meta) {
        JsonNode node = JsonUtils.fromJson(meta);
        ObjectNode obj = (ObjectNode) node.get("meta");
        obj.remove("NeuralNetClasses");
        obj.remove("NNetFeatures");
        return JsonUtils.toJson(node);
    }
}
