package ru.yandex.direct.avatars.client.model;

import java.util.Objects;

/**
 * Объект однозначно идентифицируроющий картинку в МДС. Для идентификации картинки нужны все значения: неймспейс, идентификатор группы, и ключ.
 */
public class AvatarId {
    /**
     * Неймспейс (конфиг) в аватарнице МДС.
     */
    private final String namespace;
    /**
     * Группа серверов на которых хранится картинка.
     */
    private final int groupId;
    /**
     * Ключ по которому отдаётся картинка.
     * В документации Аватарницы это же поле где-то зовётся именем картинки (imagename), а где-то идентификатором (id).
     */
    private final String key;

    public AvatarId(String namespace, int groupId, String key) {
        this.namespace = namespace;
        this.groupId = groupId;
        this.key = key;
    }

    public static AvatarId fromIdString(String avatarPath) {
        String[] split = avatarPath.split("/");
        if (split.length != 3) {
            throw new IllegalArgumentException(String.format(
                    "'%s' is a illegal path of avatar. It must have such format: $namespace/$group-id/$key.",
                    avatarPath));
        }
        return new AvatarId(split[0], Integer.parseInt(split[1]), split[2]);
    }

    public String getNamespace() {
        return namespace;
    }

    public int getGroupId() {
        return groupId;
    }

    public String getKey() {
        return key;
    }

    @Override
    public String toString() {
        return toAvatarIdString();
    }

    public String toAvatarIdString() {
        return namespace + "/" + Integer.toString(groupId) + "/" + key;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        AvatarId avatarId = (AvatarId) o;
        return groupId == avatarId.groupId &&
                namespace.equals(avatarId.namespace) &&
                key.equals(avatarId.key);
    }

    @Override
    public int hashCode() {
        return Objects.hash(namespace, groupId, key);
    }
}
