package ru.yandex.direct.balance.client;

import java.math.BigDecimal;
import java.time.Duration;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.balance.client.exception.BalanceClientException;
import ru.yandex.direct.balance.client.model.method.BalanceClassMethodSpec;
import ru.yandex.direct.balance.client.model.method.BalanceExtendedStatusMethodSpec;
import ru.yandex.direct.balance.client.model.method.BalanceExtendedUntypedStatusMethodSpec;
import ru.yandex.direct.balance.client.model.method.BalanceSimpleStatusMethodSpec;
import ru.yandex.direct.balance.client.model.method.BalanceTsvMethodSpec;
import ru.yandex.direct.balance.client.model.request.CheckBindingRequest;
import ru.yandex.direct.balance.client.model.request.CreateClientRequest;
import ru.yandex.direct.balance.client.model.request.CreateInvoiceRequest;
import ru.yandex.direct.balance.client.model.request.CreateOrUpdateOrdersBatchRequest;
import ru.yandex.direct.balance.client.model.request.CreatePersonRequest;
import ru.yandex.direct.balance.client.model.request.CreateRequest2Item;
import ru.yandex.direct.balance.client.model.request.CreateRequest2Request;
import ru.yandex.direct.balance.client.model.request.CreateUserClientAssociationRequest;
import ru.yandex.direct.balance.client.model.request.EditPassportRequest;
import ru.yandex.direct.balance.client.model.request.FindClientRequest;
import ru.yandex.direct.balance.client.model.request.GetBankRequest;
import ru.yandex.direct.balance.client.model.request.GetCardBindingURLRequest;
import ru.yandex.direct.balance.client.model.request.GetClientNdsRequest;
import ru.yandex.direct.balance.client.model.request.GetClientPersonsRequest;
import ru.yandex.direct.balance.client.model.request.GetFirmCountryCurrencyRequest;
import ru.yandex.direct.balance.client.model.request.GetLinkedClientsRequest;
import ru.yandex.direct.balance.client.model.request.GetManagersInfoRequest;
import ru.yandex.direct.balance.client.model.request.GetOverdraftParamsRequest;
import ru.yandex.direct.balance.client.model.request.GetPartnerContractsRequest;
import ru.yandex.direct.balance.client.model.request.GetPassportByUidRequest;
import ru.yandex.direct.balance.client.model.request.GetRequestChoicesRequest;
import ru.yandex.direct.balance.client.model.request.ListClientPassportsRequest;
import ru.yandex.direct.balance.client.model.request.ListPaymentMethodsSimpleRequest;
import ru.yandex.direct.balance.client.model.request.ModulusReminderRequest;
import ru.yandex.direct.balance.client.model.request.PayRequestRequest;
import ru.yandex.direct.balance.client.model.request.RemoveUserClientAssociationRequest;
import ru.yandex.direct.balance.client.model.request.SetAgencyLimitedRepSubclientsRequest;
import ru.yandex.direct.balance.client.model.request.SetOverdraftParamsRequest;
import ru.yandex.direct.balance.client.model.request.TearOffPromocodeRequest;
import ru.yandex.direct.balance.client.model.request.createtransfermultiple.CreateTransferMultipleRequest;
import ru.yandex.direct.balance.client.model.response.BalanceBankDescription;
import ru.yandex.direct.balance.client.model.response.BalanceExtendedStatusResponse;
import ru.yandex.direct.balance.client.model.response.CheckBindingResponse;
import ru.yandex.direct.balance.client.model.response.ClientNdsItem;
import ru.yandex.direct.balance.client.model.response.ClientPassportInfo;
import ru.yandex.direct.balance.client.model.response.CreateRequest2Response;
import ru.yandex.direct.balance.client.model.response.DirectDiscountItem;
import ru.yandex.direct.balance.client.model.response.FindClientResponseItem;
import ru.yandex.direct.balance.client.model.response.FirmCountryCurrencyItem;
import ru.yandex.direct.balance.client.model.response.GetCardBindingURLResponse;
import ru.yandex.direct.balance.client.model.response.GetClientPersonsResponseItem;
import ru.yandex.direct.balance.client.model.response.GetOverdraftParamsResponse;
import ru.yandex.direct.balance.client.model.response.GetRequestChoicesResponse;
import ru.yandex.direct.balance.client.model.response.GetRequestChoicesResponseFull;
import ru.yandex.direct.balance.client.model.response.LinkedClientsItem;
import ru.yandex.direct.balance.client.model.response.ListPaymentMethodsSimpleResponseItem;
import ru.yandex.direct.balance.client.model.response.ManagerInfoMapResponse;
import ru.yandex.direct.balance.client.model.response.PartnerContractInfo;
import ru.yandex.direct.balance.client.model.response.PayRequestResponse;
import ru.yandex.direct.balance.client.model.response.ProcessedInvoiceItem;

import static java.util.Arrays.asList;
import static java.util.Collections.emptyMap;

/**
 * Клиент для основного API баланса (описание методов: https://wiki.yandex-team.ru/balance/xmlrpc/)
 * <p>
 * Пример использования:
 * <pre> {@code
 *     BalanceXmlRpcClientConfig config = new BalanceXmlRpcClientConfig()
 *         .withServerUrl(new URL("http://greed-ts.paysys.yandex.ru:8002/xmlrpc"));
 *     BalanceXmlRpcClient balanceXmlRpcClient = new BalanceXmlRpcClient(config);
 *     BalanceClient client = new BalanceClient(balanceXmlRpcClient);
 *
 *     // Простой метод
 *     List<DirectBrandItem> directBrands = client.getDirectBrand(2, 1, Duration.ofSeconds(30));
 *     doSomething(directBrands);
 *
 *     // Метод посложнее
 *     CreateClientRequest request = new CreateClientRequest()
 *             .withOperatorUid(1234)
 *             .with...;
 *     request.setTimeout(Duration.ofSeconds(60));
 *     request.setMaxRetries(3);
 *     long clientId = client.createClient(request);
 *     doSomethingElse(clientId);
 * }</pre>
 */
@ParametersAreNonnullByDefault
public class BalanceClient {
    /**
     * Мы используем Balance2.*, потому что это модно, современно и соответствует спеку (в отличие от Balance.*)
     */
    private static final BalanceTsvMethodSpec<LinkedClientsItem> GET_LINKED_CLIENTS =
            new BalanceTsvMethodSpec<>("Balance2.GetLinkedClients", LinkedClientsItem.class);
    private static final BalanceTsvMethodSpec<DirectDiscountItem> GET_DIRECT_DISCOUNT =
            new BalanceTsvMethodSpec<>("Balance2.GetDirectDiscount", DirectDiscountItem.class);
    private static final BalanceTsvMethodSpec<ClientNdsItem> GET_CLIENT_NDS =
            new BalanceTsvMethodSpec<>("Balance2.GetClientNDS", ClientNdsItem.class);
    private static final BalanceClassMethodSpec<BalanceBankDescription> GET_BANK =
            new BalanceClassMethodSpec<>("Balance2.GetBank", BalanceBankDescription.class);
    private static final BalanceExtendedStatusMethodSpec<FirmCountryCurrencyItem[]> GET_FIRM_COUNTRY_CURRENCY =
            new BalanceExtendedStatusMethodSpec<>("Balance2.GetFirmCountryCurrency",
                    FirmCountryCurrencyItem[].class);
    static final BalanceExtendedStatusMethodSpec<Integer> CREATE_CLIENT =
            new BalanceExtendedStatusMethodSpec<>("Balance2.CreateClient", Integer.class);
    static final BalanceSimpleStatusMethodSpec CREATE_USER_CLIENT_ASSOCIATION =
            new BalanceSimpleStatusMethodSpec("Balance2.CreateUserClientAssociation");
    static final BalanceClassMethodSpec<CreateRequest2Response> CREATE_REQUEST_2 =
            new BalanceClassMethodSpec<>("Balance2.CreateRequest2", CreateRequest2Response.class);
    static final BalanceClassMethodSpec<PayRequestResponse> PAY_REQUEST =
            new BalanceClassMethodSpec<>("Balance2.PayRequest", PayRequestResponse.class);
    private static final BalanceClassMethodSpec<Long> CREATE_INVOICE =
            new BalanceClassMethodSpec<>("Balance2.CreateInvoice", Long.class);
    static final BalanceClassMethodSpec<CheckBindingResponse> CHECK_BINDING =
            new BalanceClassMethodSpec<>("Balance2.CheckBinding", CheckBindingResponse.class);
    static final BalanceClassMethodSpec<GetClientPersonsResponseItem[]> GET_CLIENT_PERSONS =
            new BalanceClassMethodSpec<>("Balance2.GetClientPersons", GetClientPersonsResponseItem[].class);
    static final BalanceClassMethodSpec<Long> CREATE_PERSON =
            new BalanceClassMethodSpec<>("Balance2.CreatePerson", Long.class);
    static final BalanceClassMethodSpec<GetCardBindingURLResponse> GET_CARD_BINDING_URL =
            new BalanceClassMethodSpec<>("Balance2.GetCardBindingURL", GetCardBindingURLResponse.class);
    private static final BalanceSimpleStatusMethodSpec REMOVE_USER_CLIENT_ASSOCIATION =
            new BalanceSimpleStatusMethodSpec("Balance2.RemoveUserClientAssociation");
    private static final BalanceClassMethodSpec<ClientPassportInfo> EDIT_PASSPORT_INFO =
            new BalanceClassMethodSpec<>("Balance2.EditPassport", ClientPassportInfo.class);
    static final BalanceClassMethodSpec<ClientPassportInfo[]> LIST_CLIENT_PASSPORTS =
            new BalanceClassMethodSpec<>("Balance2.ListClientPassports", ClientPassportInfo[].class);
    private static final BalanceClassMethodSpec<PartnerContractInfo[]> GET_PARTNER_CONTRACTS =
            new BalanceClassMethodSpec<>("Balance2.GetPartnerContracts", PartnerContractInfo[].class);
    private static final BalanceExtendedStatusMethodSpec<ProcessedInvoiceItem[]> TEAR_OFF_PROMOCODE =
            new BalanceExtendedStatusMethodSpec<>("Balance2.TearOffPromocode", ProcessedInvoiceItem[].class);
    static final BalanceClassMethodSpec<GetRequestChoicesResponse> GET_REQUEST_CHOICES =
            new BalanceClassMethodSpec<>("Balance2.GetRequestChoices", GetRequestChoicesResponse.class);
    private static final BalanceClassMethodSpec<GetRequestChoicesResponseFull> GET_REQUEST_CHOICES_FULL =
            new BalanceClassMethodSpec<>("Balance2.GetRequestChoices", GetRequestChoicesResponseFull.class);
    static final BalanceClassMethodSpec<GetOverdraftParamsResponse> GET_OVERDRAFT_PARAMS =
            new BalanceClassMethodSpec<>("Balance2.GetOverdraftParams", GetOverdraftParamsResponse.class);
    private static final BalanceSimpleStatusMethodSpec SET_OVERDRAFT_PARAMS =
            new BalanceSimpleStatusMethodSpec("Balance2.SetOverdraftParams");
    private static final BalanceExtendedUntypedStatusMethodSpec<FindClientResponseItem[]> FIND_CLIENT =
            new BalanceExtendedUntypedStatusMethodSpec<>("Balance2.FindClient", FindClientResponseItem[].class);
    private static final BalanceClassMethodSpec<ListPaymentMethodsSimpleResponseItem> LIST_PAYMENT_METHODS_SIMPLE =
            new BalanceClassMethodSpec<>("Balance2.ListPaymentMethodsSimple",
                    ListPaymentMethodsSimpleResponseItem.class);
    static final BalanceClassMethodSpec<ListPaymentMethodsSimpleResponseItem> LIST_PAYMENT_METHODS =
            new BalanceClassMethodSpec<>("BalanceSimple.ListPaymentMethods",
                    ListPaymentMethodsSimpleResponseItem.class);
    private static final BalanceClassMethodSpec<ClientPassportInfo> GET_PASSPORT_BY_UID =
            new BalanceClassMethodSpec<>("Balance2.GetPassportByUid", ClientPassportInfo.class);
    private static final BalanceClassMethodSpec<ManagerInfoMapResponse> GET_MANAGERS_INFO =
            new BalanceClassMethodSpec<>("Balance2.GetManagersInfo", ManagerInfoMapResponse.class);
    static final BalanceClassMethodSpec<String[][]> CREATE_OR_UPDATE_ORDERS_BATCH =
            new BalanceClassMethodSpec<>("Balance2.CreateOrUpdateOrdersBatch", String[][].class);
    private static final BalanceClassMethodSpec<Object[]> CREATE_TRANSFER_MULTIPLE =
            new BalanceClassMethodSpec<>("Balance2.CreateTransferMultiple", Object[].class);

    private final BalanceXmlRpcClient balance;
    private final BalanceXmlRpcClient balanceSimple;

    public BalanceClient(BalanceXmlRpcClient balance, BalanceXmlRpcClient balanceSimple) {
        this.balance = balance;
        this.balanceSimple = balanceSimple;
    }

    /**
     * Создаёт клиента или обновляет существующего.
     * <p>
     * https://wiki.yandex-team.ru/balance/xmlrpc/#balance.createclient
     *
     * @return идентификатор клиента (ClientId)
     */
    public long createClient(CreateClientRequest request) {
        Objects.requireNonNull(request.getOperatorUid(), "operatorUid");

        return balance.call(BalanceClient.CREATE_CLIENT, request).getData().longValue();
    }

    /**
     * Создаёт клиента или обновляет существующего.
     * <p>
     * https://wiki.yandex-team.ru/balance/xmlrpc/#balance.createclient
     *
     * @param operatorUid UID оператора
     * @return идентификатор клиента (ClientId)
     */
    public long createClient(long operatorUid, CreateClientRequest request) {
        request.setOperatorUid(operatorUid);

        return createClient(request);
    }

    /**
     * Создает заготовку счета
     * https://wiki.yandex-team.ru/balance/xmlrpc/#balance.createrequest2
     */
    public CreateRequest2Response createRequest(Long operatorUid, Long clientId, CreateRequest2Item createRequest2Item,
                                                boolean forceUnmoderated, @Nullable String promocode, boolean denyPromocode) {
        Objects.requireNonNull(operatorUid, "operatorUid");
        Objects.requireNonNull(clientId, "clientId");
        Objects.requireNonNull(createRequest2Item, "createRequest2Item");

        CreateRequest2Request request = new CreateRequest2Request()
                .withOperatorUid(operatorUid)
                .withClientId(clientId)
                .withItems(List.of(createRequest2Item))
                .withPromocode(promocode)
                .withForceUnmoderated(forceUnmoderated)
                .withDenyPromocode(denyPromocode);

        return balance.call(BalanceClient.CREATE_REQUEST_2, request);
    }

    /**
     * Инициализация процесса оплаты реквеста
     * https://wiki.yandex-team.ru/balance/xmlrpc/#balance.payrequest
     */
    public PayRequestResponse payRequest(PayRequestRequest request) {
        return balance.call(BalanceClient.PAY_REQUEST, request);
    }

    /**
     * Создает заготовку счета
     * https://wiki.yandex-team.ru/balance/xmlrpc/#balance.createinvoice
     */
    public Long createInvoice(CreateInvoiceRequest request) {
        return balance.call(BalanceClient.CREATE_INVOICE, request);
    }

    public CheckBindingResponse checkBinding(CheckBindingRequest request) {
        return balance.call(BalanceClient.CHECK_BINDING, request);
    }

    public List<GetClientPersonsResponseItem> getClientPersons(Long clientId) {
        GetClientPersonsRequest request = new GetClientPersonsRequest().withClientId(clientId);
        return asList(balance.call(BalanceClient.GET_CLIENT_PERSONS, request));
    }

    public Long createPerson(CreatePersonRequest request) {
        return balance.call(CREATE_PERSON, request);
    }

    public GetCardBindingURLResponse getCardBindingURL(Long operatorUid, String currencyCode,
                                                       @Nullable String returnPath,
                                                       Integer serviceId, boolean isMobileForm) {

        GetCardBindingURLRequest.GetCardBindingURLFormType formType =
                new GetCardBindingURLRequest.GetCardBindingURLFormType(isMobileForm);

        GetCardBindingURLRequest request = new GetCardBindingURLRequest()
                .withCurrency(currencyCode)
                .withReturnPath(returnPath)
                .withServiceID(serviceId)
                .withOperatorUid(operatorUid)
                .withFormType(formType);

        return balance.call(BalanceClient.GET_CARD_BINDING_URL, request);
    }

    /**
     * Создает представителя клиента
     * <p>
     * https://wiki.yandex-team.ru/balance/xmlrpc/#balance.createuserclientassociation
     *
     * @param operatorUid       UID оператора
     * @param clientId          Идентификатор клиента
     * @param representativeUid UID представителя
     */
    public void createUserClientAssociation(Long operatorUid, Long clientId, Long representativeUid) {
        Objects.requireNonNull(operatorUid, "operatorUid");
        Objects.requireNonNull(clientId, "clientId");
        Objects.requireNonNull(representativeUid, "representativeUid");

        CreateUserClientAssociationRequest request = new CreateUserClientAssociationRequest()
                .withOperatorUid(operatorUid)
                .withClientId(clientId)
                .withRepresentativeUid(representativeUid);
        balance.call(CREATE_USER_CLIENT_ASSOCIATION, request);
    }

    /**
     * Удаляет представителя клиента
     * <p>
     * https://wiki.yandex-team.ru/balance/xmlrpc/#balance.removeuserclientassociation
     *
     * @param operatorUid       UID оператора
     * @param clientId          Идентификатор клиента
     * @param representativeUid UID представителя
     */
    public void removeUserClientAssociation(Long operatorUid, Long clientId, Long representativeUid) {
        Objects.requireNonNull(operatorUid, "operatorUid");
        Objects.requireNonNull(clientId, "clientId");
        Objects.requireNonNull(representativeUid, "representativeUid");

        RemoveUserClientAssociationRequest request = new RemoveUserClientAssociationRequest()
                .withOperatorUid(operatorUid)
                .withClientId(clientId)
                .withRepresentativeUid(representativeUid);
        balance.call(REMOVE_USER_CLIENT_ASSOCIATION, request);
    }

    /**
     * Назначить субклиентов ограниченному представителю агентства
     * чтобы ограниченный представитель видел в балансе только своих клиентов
     *
     * @param operatorUid       UID оператора
     * @param clientId          clientId агентства
     * @param representativeUid UID представителя агентства
     * @param subclientIds      список ClientID субклиентов, к которым представитель имеет доступ
     */
    public void setAgencyLimitedRepSubclients(Long operatorUid, Long clientId, Long representativeUid,
                                              List<Long> subclientIds) {
        Objects.requireNonNull(operatorUid, "operatorUid");
        Objects.requireNonNull(clientId, "clientId");
        Objects.requireNonNull(representativeUid, "representativeUid");
        Objects.requireNonNull(subclientIds, "subclientIds");

        CreateUserClientAssociationRequest request = new SetAgencyLimitedRepSubclientsRequest()
                .withSubclientIds(subclientIds)
                .withOperatorUid(operatorUid)
                .withClientId(clientId)
                .withRepresentativeUid(representativeUid);
        balance.call(CREATE_USER_CLIENT_ASSOCIATION, request);
    }

    /**
     * Изменяет ползоваетельский лимит овердрафта
     * <br>
     * Пример:
     * HOST = 'https://xmlrpc-auto-overdraft.greed-branch.paysys.yandex.ru/xmlrpc'
     * server = xmlrpclib.ServerProxy(HOST, allow_none=1)
     * <p>
     * params = dict(
     * PersonID=7441519,
     * ServiceID=7,
     * PaymentMethodCC='bank',
     * IsoCurrency='RUB',
     * ClientLimit=1000,
     * )
     * server.Balance.SetOverdraftParams(params)
     *
     * @param personId          ID плательщика
     * @param serviceId         ID сервиса в балансе (для Директа - 7)
     * @param paymentMethodCode код способа оплаты
     * @param isoCurrencyCode   ISO код валюты
     * @param clientLimit       клиентский лимит овердрафта на ОС
     */
    public void setOverdraftParams(Long personId, Integer serviceId, String paymentMethodCode, String isoCurrencyCode,
                                   BigDecimal clientLimit) {
        Objects.requireNonNull(personId, "operatorUid");
        Objects.requireNonNull(paymentMethodCode, "paymentMethodCode");
        Objects.requireNonNull(isoCurrencyCode, "isoCurrencyCode");
        Objects.requireNonNull(clientLimit, "clientLimit");

        SetOverdraftParamsRequest request = new SetOverdraftParamsRequest()
                .withPersonId(personId)
                .withServiceId(serviceId)
                .withPaymentMethodCode(paymentMethodCode)
                .withIsoCurrencyCode(isoCurrencyCode)
                .withClientLimit(clientLimit);
        balance.call(SET_OVERDRAFT_PARAMS, request);
    }

    /**
     * Отредактировать паспортную информацию о представителе
     * <p>
     * https://wiki.yandex-team.ru/balance/xmlrpc/#balance.editpassport
     *
     * @param operatorUid  UID оператора
     * @param clientUid    UID клиента
     * @param passportInfo Новые параметры
     */
    public ClientPassportInfo editPassport(Long operatorUid, Long clientUid, ClientPassportInfo passportInfo) {
        Objects.requireNonNull(operatorUid, "operatorUid");
        Objects.requireNonNull(clientUid, "clientUid");
        Objects.requireNonNull(passportInfo, "request");

        EditPassportRequest request = new EditPassportRequest()
                .withOperatorUid(operatorUid)
                .withClientUid(clientUid)
                .withClientPassportInfo(passportInfo);
        return balance.call(EDIT_PASSPORT_INFO, request);
    }

    /**
     * Получение списка доступных способов оплаты
     * <p>
     * https://wiki.yandex-team.ru/balance/xmlrpc/paymentviatrustprocess/#balance.getrequestchoices
     *
     * @param operatorUid UID оператора
     * @param clientId    ID клиента
     */
    public GetRequestChoicesResponse getPaymentMethods(Long operatorUid, Long clientId,
                                                       List<CreateRequest2Item> orderItems) {
        var getRequestChoicesRequest = createGetRequestChoicesRequest(operatorUid, clientId, orderItems);
        return balance.call(GET_REQUEST_CHOICES, getRequestChoicesRequest);
    }

    /**
     * Получение детального списка доступных способов оплаты
     * <p>
     * https://wiki.yandex-team.ru/balance/xmlrpc/paymentviatrustprocess/#balance.getrequestchoices
     *
     * @param operatorUid UID оператора
     * @param clientId    ID клиента
     */
    public GetRequestChoicesResponseFull getPaymentMethodsFull(Long operatorUid, Long clientId,
                                                               List<CreateRequest2Item> orderItems) {
        var getRequestChoicesRequest = createGetRequestChoicesRequest(operatorUid, clientId, orderItems);
        return balance.call(GET_REQUEST_CHOICES_FULL, getRequestChoicesRequest);
    }

    private GetRequestChoicesRequest createGetRequestChoicesRequest(Long operatorUid, Long clientId,
                                                                    List<CreateRequest2Item> orderItems) {
        Objects.requireNonNull(operatorUid, "operatorUid");
        Objects.requireNonNull(clientId, "clientId");
        Objects.requireNonNull(orderItems, "orderItems");

        return new GetRequestChoicesRequest()
                .withOperatorUid(operatorUid)
                .withClientId(clientId)
                .withItems(orderItems);
    }

    /**
     * Получение настроек автоовердрафта
     *
     * @param clientId ID клиента
     */
    public GetOverdraftParamsResponse getOverdraftParams(Integer serviceId, Long clientId) {
        Objects.requireNonNull(clientId, "clientId");

        GetOverdraftParamsRequest request = new GetOverdraftParamsRequest()
                .withServiceId(serviceId)
                .withClientId(clientId);
        return balance.call(GET_OVERDRAFT_PARAMS, request);
    }

    /**
     * Получить паспортную информацию о представителях клиента
     * <p>
     * https://wiki.yandex-team.ru/balance/xmlrpc/#balance.listclientpassports
     *
     * @param operatorUid UID оператора
     * @param clientId    Идентификатор клиента
     */
    public List<ClientPassportInfo> getClientRepresentativePassports(Long operatorUid, Long clientId) {
        Objects.requireNonNull(operatorUid, "operatorUid");
        Objects.requireNonNull(clientId, "clientId");

        ListClientPassportsRequest request = new ListClientPassportsRequest()
                .withOperatorUid(operatorUid)
                .withClientId(clientId);
        return asList(balance.call(LIST_CLIENT_PASSPORTS, request));
    }

    /**
     * Получить контракты, плательщиков и доп. соглашения клиента
     * <p>
     *
     * @param clientId           ID клиента в балансе
     * @param externalContractId внешний номер договора, например, "285245/18"
     * @param timeout            https://st.yandex-team.ru/DIRECT-81789
     * @link https://wiki.yandex-team.ru/partner/w/external-services-api-balance-get-partner-contracts/
     */
    public List<PartnerContractInfo> getPartnerContracts(@Nullable Long clientId, @Nullable String externalContractId,
                                                         Duration timeout) {
        GetPartnerContractsRequest request = new GetPartnerContractsRequest(clientId, externalContractId);
        request.setTimeout(timeout);
        return asList(balance.call(GET_PARTNER_CONTRACTS, request));
    }

    /**
     * Метод возвращает для клиента + агентства, по их привязке к определённой стране, саму страну,
     * фирму обслуживания и разрешённые для него валюты с указанием флажка "нерезидент".
     * <p>
     * https://wiki.yandex-team.ru/balance/xmlrpc/#balance.getfirmcountrycurrency
     */
    @Nonnull
    public List<FirmCountryCurrencyItem> getFirmCountryCurrency(GetFirmCountryCurrencyRequest request) {
        BalanceExtendedStatusResponse<FirmCountryCurrencyItem[]> response =
                balance.call(BalanceClient.GET_FIRM_COUNTRY_CURRENCY, request);
        return asList(response.getData());
    }

    /**
     * Отдает по всем клиентам, у которых установлен region_id, информацию о действующей ставке НДС
     * с указанием периода времени.
     * <p>
     * Запрос передаётся с полями serviceId, mod и rem. Этими ключами можно регулировать размер пачки
     * данных на выходе. Возвращаются данные по всем клиентам, для которых (clientId % mod) == rem.
     * <p>
     * https://wiki.yandex-team.ru/balance/xmlrpc/#balance.getclientnds
     */
    @Nonnull
    public List<ClientNdsItem> getClientNds(GetClientNdsRequest request) {
        return balance.call(BalanceClient.GET_CLIENT_NDS, request);
    }

    /**
     * Отдает информацию о действующей ставке НДС для указанного клиента, если у него установлен region_id
     * <p>
     * https://wiki.yandex-team.ru/balance/xmlrpc/#balance.getclientnds
     *
     * @param serviceId ID сервиса в балансе (для Директа - 7)
     * @param clientId  Идентификатор клиента
     */
    @Nonnull
    public List<ClientNdsItem> getClientNds(int serviceId, int clientId) {
        return getClientNds(new GetClientNdsRequest()
                .withServiceId(serviceId)
                .withModulus(Integer.MAX_VALUE)
                .withReminder(clientId));
    }

    /**
     * Отдает по всем клиентам информацию о рассчитанных скидках на Директ.
     * <p>
     * https://wiki.yandex-team.ru/balance/xmlrpc/#balance.getdirectdiscount
     */
    @Nonnull
    public List<DirectDiscountItem> getDirectDiscount(ModulusReminderRequest request) {
        return balance.call(BalanceClient.GET_DIRECT_DISCOUNT, request);
    }

    /**
     * Отдает по всем клиентам информацию о рассчитанных скидках на Директ.
     * <p>
     * https://wiki.yandex-team.ru/balance/xmlrpc/#balance.getdirectdiscount
     *
     * @param modulus  делитель
     * @param reminder остаток от деления ClientId на делитель
     * @param timeout  значение таимаута для запроса
     */
    @Nonnull
    public List<DirectDiscountItem> getDirectDiscount(int modulus, int reminder, Duration timeout) {
        ModulusReminderRequest request = new ModulusReminderRequest().withModulus(modulus).withReminder(reminder);
        request.setTimeout(timeout);
        return getDirectDiscount(request);
    }


    /**
     * Отдает по всем клиентам информацию о рассчитанных скидках на Директ.
     * <p>
     * https://wiki.yandex-team.ru/balance/xmlrpc/#balance.getdirectdiscount
     *
     * @param modulus  делитель
     * @param reminder остаток от деления ClientId на делитель
     */
    @Nonnull
    public List<DirectDiscountItem> getDirectDiscount(int modulus, int reminder) {
        ModulusReminderRequest request = new ModulusReminderRequest().withModulus(modulus).withReminder(reminder);
        return getDirectDiscount(request);
    }

    /**
     * Отдает информацию о всех связанных клиентах.
     * <p>
     * Запрос передается со списком типов связей, которые нужно получить из ручки
     * <p>
     * https://wiki.yandex-team.ru/balance/xmlrpc/#balance.getlinkedclients
     */
    @Nonnull
    public List<LinkedClientsItem> getLinkedClients(List<Integer> linkTypes) {
        GetLinkedClientsRequest request = new GetLinkedClientsRequest();
        request.setLinkTypes(linkTypes);
        return balance.call(BalanceClient.GET_LINKED_CLIENTS, request);
    }

    /**
     * Возвращает данные банка по БИК для РФ или SWIFT для зарубежных банков
     * <p>
     * https://wiki.yandex-team.ru/balance/xmlrpc/#balance.getbank
     *
     * @param swift значение SWIFT для зарубежных банков
     */
    public BalanceBankDescription getBank(String swift) {
        GetBankRequest request = new GetBankRequest().withSwift(swift);
        return getBank(request);
    }

    /**
     * Возвращает данные банка по БИК для РФ или SWIFT для зарубежных банков
     * <p>
     * https://wiki.yandex-team.ru/balance/xmlrpc/#balance.getbank
     *
     * @param swift   значение SWIFT для зарубежных банков
     * @param timeout значение таимаута для запроса
     */
    public BalanceBankDescription getBank(String swift, Duration timeout) {
        GetBankRequest request = new GetBankRequest().withSwift(swift);
        request.setTimeout(timeout);
        return getBank(request);
    }

    /**
     * Возвращает данные банка по БИК для РФ или SWIFT для зарубежных банков
     * <p>
     * https://wiki.yandex-team.ru/balance/xmlrpc/#balance.getbank
     */
    public BalanceBankDescription getBank(GetBankRequest request) {
        return balance.call(BalanceClient.GET_BANK, request);
    }

    /**
     * Отменить промокод по счету.
     * Требуется обязательного указания счета или клиента или заказа.
     * <p>
     * https://st.yandex-team.ru/BALANCE-28045#1526890733000
     */
    @Nonnull
    public List<ProcessedInvoiceItem> tearOffPromocode(TearOffPromocodeRequest request) {
        Objects.requireNonNull(request, "request");
        Objects.requireNonNull(request.getOperatorUid(), "operatorUid");

        try {
            return asList(balance.call(TEAR_OFF_PROMOCODE, request).getData());
        } catch (BalanceClientException e) {
            if (e.getMessage().contains("PC_TEAR_OFF_NO_FREE_CONSUMES")) {
                // на счёте оказались неотрываемые промокодные средства
                return List.of();
            }
            throw e;
        }
    }

    /**
     * Метод возвращает список клиентов по переданному запросу на поиск
     * <p>
     * https://wiki.yandex-team.ru/Balance/XmlRpc/#balance.findclient
     */
    @Nonnull
    public List<FindClientResponseItem> findClient(FindClientRequest request) {
        BalanceExtendedStatusResponse<FindClientResponseItem[]> response =
                balance.call(BalanceClient.FIND_CLIENT, request);
        return asList(response.getData());
    }

    /**
     * Запрос методов платежей доступных пользователю:
     * https://wiki.yandex-team.ru/Balance/XmlRpc/#balance.listpaymentmethodssimple
     */
    @Nonnull
    public ListPaymentMethodsSimpleResponseItem listPaymentMethodsSimple(ListPaymentMethodsSimpleRequest request) {
        return balance.call(BalanceClient.LIST_PAYMENT_METHODS_SIMPLE, request);
    }

    @Nonnull
    public ListPaymentMethodsSimpleResponseItem listPaymentMethods(ListPaymentMethodsSimpleRequest request) {
        return balanceSimple.call(BalanceClient.LIST_PAYMENT_METHODS, request);
    }

    /**
     * Запрос паспортной информации (информации о представителе) по известному Yandex.Passport UID.
     * https://wiki.yandex-team.ru/Balance/XmlRpc/#balance.getpassportbyuid
     */
    @Nonnull
    public ClientPassportInfo getPassportByUid(Long operatorUid, Long uid) {
        GetPassportByUidRequest request = new GetPassportByUidRequest()
                .withOperatorUid(operatorUid)
                .withUid(uid)
                .withRelations(emptyMap());
        return balance.call(BalanceClient.GET_PASSPORT_BY_UID, request);
    }

    /**
     * Проверка существования менеджера в Балансе. Используется ручка GetManagersInfo:
     * https://wiki.yandex-team.ru/balance/xmlrpc/#balance.getmanagersinfo
     * Возвращается список uid'ов менеджеров, которые были найдены в балансе из переданного списка.
     */
    @Nonnull
    public Set<Long> massCheckManagersExist(Collection<Long> managerUids) {
        GetManagersInfoRequest request = new GetManagersInfoRequest()
                .withManagerUids(managerUids);
        ManagerInfoMapResponse managersInfo = balance.call(BalanceClient.GET_MANAGERS_INFO, request);
        return managersInfo.entrySet().stream()
                .filter(e -> e.getValue() != null)
                .map(Map.Entry::getKey)
                .collect(Collectors.toSet());
    }

    /**
     * Пакетное создание/обновление заказов в Балансе. Используется ручка CreateOrUpdateOrdersBatch:
     * https://wiki.yandex-team.ru/balance/xmlrpc/#balance.createorupdateordersbatch
     * Возвращаемое значение: список пар (код, строка), содержащих результат для каждого создаваемого или
     * обновляемого заказа.
     */
    public List<List<String>> createOrUpdateOrdersBatch(CreateOrUpdateOrdersBatchRequest request) {
        String[][] response = balance.call(BalanceClient.CREATE_OR_UPDATE_ORDERS_BATCH, request);
        return Arrays.stream(response)
                .map(Arrays::asList)
                .collect(Collectors.toList());
    }

    public void createTransferMultiple(CreateTransferMultipleRequest request) {
        balance.call(BalanceClient.CREATE_TRANSFER_MULTIPLE, request);
    }
}
