package ru.yandex.direct.balance.client;

import java.time.Duration;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.primitives.Ints;
import org.apache.ws.commons.util.NamespaceContextImpl;
import org.apache.xmlrpc.XmlRpcException;
import org.apache.xmlrpc.client.XmlRpcClient;
import org.apache.xmlrpc.client.XmlRpcClientConfigImpl;
import org.apache.xmlrpc.client.XmlRpcHttpTransportException;
import org.apache.xmlrpc.common.TypeFactoryImpl;
import org.apache.xmlrpc.common.XmlRpcController;
import org.apache.xmlrpc.common.XmlRpcLoadException;
import org.apache.xmlrpc.common.XmlRpcStreamConfig;
import org.apache.xmlrpc.common.XmlRpcWorkerFactory;
import org.apache.xmlrpc.parser.NullParser;
import org.apache.xmlrpc.parser.TypeParser;
import org.apache.xmlrpc.serializer.NullSerializer;
import org.apache.xmlrpc.serializer.TypeSerializer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.xml.sax.SAXException;

import ru.yandex.direct.balance.client.exception.BalanceClientException;
import ru.yandex.direct.balance.client.model.method.BaseBalanceMethodSpec;
import ru.yandex.direct.balance.client.model.request.BalanceRpcRequestParam;
import ru.yandex.direct.solomon.SolomonExternalSystemMonitorService;
import ru.yandex.direct.solomon.SolomonResponseMonitorStatus;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.tracing.TraceProfile;
import ru.yandex.direct.utils.JsonUtils;
import ru.yandex.direct.utils.SystemUtils;

import static ru.yandex.direct.balance.client.BalanceClient.CHECK_BINDING;
import static ru.yandex.direct.balance.client.BalanceClient.CREATE_CLIENT;
import static ru.yandex.direct.balance.client.BalanceClient.CREATE_OR_UPDATE_ORDERS_BATCH;
import static ru.yandex.direct.balance.client.BalanceClient.CREATE_PERSON;
import static ru.yandex.direct.balance.client.BalanceClient.CREATE_REQUEST_2;
import static ru.yandex.direct.balance.client.BalanceClient.CREATE_USER_CLIENT_ASSOCIATION;
import static ru.yandex.direct.balance.client.BalanceClient.GET_CARD_BINDING_URL;
import static ru.yandex.direct.balance.client.BalanceClient.GET_CLIENT_PERSONS;
import static ru.yandex.direct.balance.client.BalanceClient.GET_OVERDRAFT_PARAMS;
import static ru.yandex.direct.balance.client.BalanceClient.GET_REQUEST_CHOICES;
import static ru.yandex.direct.balance.client.BalanceClient.LIST_CLIENT_PASSPORTS;
import static ru.yandex.direct.balance.client.BalanceClient.LIST_PAYMENT_METHODS;
import static ru.yandex.direct.balance.client.BalanceClient.PAY_REQUEST;
import static ru.yandex.direct.solomon.SolomonResponseMonitorStatus.STATUS_2XX;
import static ru.yandex.direct.solomon.SolomonResponseMonitorStatus.STATUS_4XX;
import static ru.yandex.direct.solomon.SolomonResponseMonitorStatus.STATUS_5XX;
import static ru.yandex.direct.solomon.SolomonResponseMonitorStatus.STATUS_UNKNOWN;
import static ru.yandex.direct.solomon.SolomonResponseMonitorStatus.STATUS_UNPARSABLE;

/**
 * Базовый клиент для XmlRpc API Баланса. Используется через хелперы в других классах пакета
 * <p>
 * ВАЖНО: XML-RPC API Баланса иногда нарушает спецификацию (http://xmlrpc.scripting.com/spec.html) а именно:
 * "The body of the response is a single XML structure, a <methodResponse>,
 * which can contain a single <params> which contains a single <param> which contains a single <value>."
 * <p>
 * Нарушение заключается в том, что возвращаемое значение (value) может быть не одно.
 * Текущая используемая реализация клиента из-за этого возвращает в качестве ответа только ПОСЛЕДНЕЕ <value>
 * в <param>, а все предыдущие отбрасывает.
 * <p>
 * Пока это никак не влияет на получаемые нами значения, так как отбрасываются только константные статусы,
 * а в случае возврата списка значений в Балансе используются массивы, но это нужно учитывать при разработке.
 * Сейчас основной маркер того, что возвращаемое значение может быть отброшено - слово "кортеж" в документации
 * к АПИ.
 * Например:
 * - в ручке Balance.CreateClient (https://wiki.yandex-team.ru/balance/xmlrpc/#balance.createclient) по документации
 * получаем кортеж из статуса, сообщения и ClientID. Значит ответ нужно обрабатывать так,
 * как будто мы получаем только ClientID.
 * - в ручке Balance.GetFirmCountryCurrency (https://wiki.yandex-team.ru/balance/xmlrpc/#balance.getfirmcountrycurrency)
 * по документации получаем список из статуса, сообщения и данных. Значит обрабатывать нужно все три элемента
 * списка
 * <p>
 * В сервисе Balance2 таких проблем пока замечено не было.
 * <p>
 * Тип {@link Long} не поддерживается на стороне баланса библиотекой. Поля с такой размерностью передаются строками.
 */
@ParametersAreNonnullByDefault
public class BalanceXmlRpcClient {
    private static final Logger BALANCE_CALLS_LOGGER = LoggerFactory.getLogger("BALANCE_CALLS");

    private static final Set<String> MONITOR_METHODS = Set.of(
            GET_CLIENT_PERSONS.getFullName(),
            CREATE_PERSON.getFullName(),
            LIST_PAYMENT_METHODS.getFullName(),
            GET_CARD_BINDING_URL.getFullName(),
            CREATE_REQUEST_2.getFullName(),
            PAY_REQUEST.getFullName(),
            CHECK_BINDING.getFullName(),
            CREATE_CLIENT.getFullName(),
            CREATE_OR_UPDATE_ORDERS_BATCH.getFullName(),
            CREATE_USER_CLIENT_ASSOCIATION.getFullName(),
            GET_OVERDRAFT_PARAMS.getFullName(),
            GET_REQUEST_CHOICES.getFullName(),
            LIST_CLIENT_PASSPORTS.getFullName()
    );

    private static final String EXTERNAL_SYSTEM = "balance";
    /**
     * {@code SolomonDefaultMonitorStatus#STATUS_UNKNOWN} и {@code SolomonDefaultMonitorStatus#STATUS_UNPARSABLE}
     * не инициализируем заранее: если что-то массово поменяется и будут эти статусы - то даже с потерей первой точки
     * заметим, а если очень мало (1 ошибка в несколько дней) - то и не важна она нам
     */
    private static final SolomonExternalSystemMonitorService monitorService = new SolomonExternalSystemMonitorService(
            EXTERNAL_SYSTEM,
            MONITOR_METHODS
    );

    private final XmlRpcClientConfigImpl clientConfig;
    private final XmlRpcClient client;
    private final int retriesNum;

    /**
     * Workaround for
     * "Failed to parse server's response: Unknown type: nil"
     */
    static class XmlRpcTypeNil extends TypeFactoryImpl {

        XmlRpcTypeNil(XmlRpcController pController) {
            super(pController);
        }

        public TypeParser getParser(XmlRpcStreamConfig pConfig, NamespaceContextImpl pContext, String pURI,
                                    String pLocalName) {
            if (NullSerializer.NIL_TAG.equals(pLocalName) || NullSerializer.EX_NIL_TAG.equals(pLocalName)) {
                return new NullParser();
            } else {
                return super.getParser(pConfig, pContext, pURI, pLocalName);
            }
        }

        public TypeSerializer getSerializer(XmlRpcStreamConfig pConfig, Object pObject) throws SAXException {
            if (pObject instanceof XmlRpcTypeNil) {
                return new NullSerializer();
            } else {
                return super.getSerializer(pConfig, pObject);
            }
        }
    }

    public BalanceXmlRpcClient(BalanceXmlRpcClientConfig config) {
        clientConfig = getXmlRpcClientConfig(config);
        client = new XmlRpcClient();
        client.setTypeFactory(new XmlRpcTypeNil(client));
        client.setConfig(clientConfig);

        retriesNum = config.getMaxRetries();
    }

    BalanceXmlRpcClient(BalanceXmlRpcClientConfig config, XmlRpcWorkerFactory workerFactory) {
        this(config);
        client.setWorkerFactory(workerFactory);
    }

    /**
     * Создать инстанс настроек XML-RPC клиента из настроек клиента Баланса
     */
    static XmlRpcClientConfigImpl getXmlRpcClientConfig(BalanceXmlRpcClientConfig config) {
        XmlRpcClientConfigImpl xmlRpcClientConfig = new XmlRpcClientConfigImpl();
        xmlRpcClientConfig.setServerURL(config.getServerUrl());
        xmlRpcClientConfig.setConnectionTimeout(Ints.saturatedCast(config.getConnectionTimeout().toMillis()));
        xmlRpcClientConfig.setReplyTimeout(Ints.saturatedCast(config.getRequestTimeout().toMillis()));
        return xmlRpcClientConfig;
    }

    /**
     * Выполнить обращение в XML-RPC Баланса и смаппить ответ в соответствующую методу структуру
     */
    public <R, T extends BalanceRpcRequestParam> R call(BaseBalanceMethodSpec<R> methodSpec, T requestParam) {
        Object response = retriedCall(methodSpec.getFullName(), requestParam.asArray(), requestParam.getTimeout(),
                requestParam.getMaxRetries());
        try {
            R result = methodSpec.convertResponse(response);
            countSuccessfulRequest(methodSpec.getFullName());
            return result;
        } catch (Exception e) {
            countException(methodSpec.getFullName(), e);
            BALANCE_CALLS_LOGGER.error("Failed to convert balance call result", e);
            throw new BalanceClientException("Failed to convert balance call result", e);
        }
    }

    /**
     * Выполнить серию обращений к методу XML-RPC Баланса с заданным таимаутом на одно обращение и максимальным
     * количеством повторов. Вернуть результат первого успешного обращения.
     * <p>
     * Если таимаут или максимальное количество попыток не заданы, берутся значения по-умолчанию
     */
    Object retriedCall(String methodName, Object[] params, @Nullable Duration overrideTimeout,
                       @Nullable Integer overrideRetries) {
        int maxRetries = overrideRetries == null ? retriesNum : overrideRetries;
        try (TraceProfile ignore = Trace.current().profile("balance:call", methodName)) {
            for (int currentRetry = 1; ; currentRetry++) {
                logRequestOrResponseData(methodName, "request", params, currentRetry);
                try {
                    // успешные запросы считаем выше по стеку
                    // так как ошибка может быть упакована в тело ответа
                    return rawCall(methodName, params, overrideTimeout);
                } catch (XmlRpcLoadException e) {
                    // кажется, что это мёртвая ветка кода, отпилить в DIRECT-122630
                    countException(methodName, e);
                    String msg = String.format("%s call error on try %s: %s", methodName, currentRetry, e.getMessage());
                    if (currentRetry < maxRetries) {
                        BALANCE_CALLS_LOGGER.warn(msg, e);
                    } else {
                        BALANCE_CALLS_LOGGER.error(msg, e);
                        throw new BalanceClientException(msg, e);
                    }
                } catch (XmlRpcHttpTransportException e) {
                    countException(methodName, e);
                    String msg = String.format("%s call HTTP error on try %s: %s %s", methodName, currentRetry,
                            e.getStatusCode(), e.getStatusMessage());
                    if (currentRetry < maxRetries) {
                        BALANCE_CALLS_LOGGER.warn(msg, e);
                    } else {
                        BALANCE_CALLS_LOGGER.error(msg, e);
                        throw new BalanceClientException(msg, e);
                    }
                } catch (XmlRpcException e) {
                    countException(methodName, e);
                    String msg = String.format("%s call fault on try %s: %s", methodName, currentRetry, e.getMessage());
                    if (currentRetry < maxRetries) {
                        BALANCE_CALLS_LOGGER.warn(msg, e);
                    } else {
                        BALANCE_CALLS_LOGGER.error(msg, e);
                        throw new BalanceClientException(msg, e);
                    }
                }
            }
        }
    }

    /**
     * Выполнить непосредственное обращение к методу XML-RPC Баланса с заданным таимаутом и вернуть ответ
     * <p>
     * Если таимаут не задан, берется таимаут по-умолчанию
     */
    Object rawCall(String methodName, Object[] params, @Nullable Duration timeout) throws XmlRpcException {
        Object response;
        if (timeout == null || clientConfig.getReplyTimeout() == timeout.toMillis()) {
            response = client.execute(methodName, params);
        } else {
            XmlRpcClientConfigImpl newConfig = clientConfig.cloneMe();
            newConfig.setReplyTimeout(Ints.saturatedCast(timeout.toMillis()));
            response = client.execute(newConfig, methodName, params);
        }

        logRequestOrResponseData(methodName, "response", response, null);
        return response;
    }

    private void logRequestOrResponseData(String methodName, String dataType, Object dataToLog,
                                          @Nullable Integer tryNumber) {
        if (BALANCE_CALLS_LOGGER.isInfoEnabled()) {
            String tryPart = tryNumber == null ? "" : ",try=" + tryNumber;
            BALANCE_CALLS_LOGGER.info("[pid={},reqid={},method={},data_type={}{}]\t{}",
                    SystemUtils.getPid(),
                    Trace.current().getSpanId(),
                    methodName,
                    dataType,
                    tryPart,
                    JsonUtils.toJson(dataToLog));
        }
    }

    static SolomonResponseMonitorStatus guessStatusByException(Exception exception) {
        if (exception instanceof XmlRpcHttpTransportException) {
            int code = ((XmlRpcHttpTransportException) exception).getStatusCode() / 100;
            if (code == 4) {
                return STATUS_4XX;
            }
            if (code == 5) {
                return STATUS_5XX;
            }
            return STATUS_UNKNOWN;
        }

        if (exception instanceof XmlRpcException) {
            String message = exception.getMessage();
            if (message == null) {
                return STATUS_UNKNOWN;
            }
            if (message.startsWith("Error: ")
                    // считаем что это балансер снял анонс, а не нам сеть отрубили
                    || message.contains("No route to host")
                    || message.startsWith("Failed to ")
            ) {
                return STATUS_5XX;
            }

            String prefix = "<error><msg>";
            if (message.startsWith(prefix)) {
                if (message.startsWith("Error in balance payments api call:", prefix.length())) {
                    // пока не встречалось 4хх ошибок с таким префиксом
                    return STATUS_5XX;
                }
                if (message.startsWith("Error in trust api call:", prefix.length())) {
                    if (message.contains("has too many active bindings")) {
                        return STATUS_4XX;
                    } else {
                        return STATUS_5XX;
                    }
                }
                if (message.startsWith("Invalid parameter for function", prefix.length())) {
                    return STATUS_4XX;
                }
                if (message.contains("<code>ORDERS_NOT_SYNCHRONIZED</code>")
                        || message.contains("<code>PERMISSION_DENIED</code>")
                        || message.contains("<code>INVALID_PERSON_TYPE</code>")
                        || message.contains("PC_TEAR_OFF_NO_FREE_CONSUMES")
                ) {
                    return STATUS_4XX;
                }
                if (message.contains("<code>REQUEST_IS_LOCKED</code>")) {
                    return STATUS_5XX;
                }
                // неизвестные ошибки будем считать серверными
                return STATUS_5XX;
            }
        }

        if (exception instanceof BalanceClientException) {
            // расчет на то, что сюда попадаем при ошибках парсинга
            return STATUS_UNPARSABLE;
        }

        return STATUS_UNKNOWN;
    }

    private static void countSuccessfulRequest(String method) {
        monitorService.write(method, STATUS_2XX);
    }

    private static void countException(String method, Exception exception) {
        var status = guessStatusByException(exception);
        monitorService.write(method, status);
    }
}
