package ru.yandex.direct.balance.client.model.response;

import java.math.BigDecimal;
import java.time.LocalDate;
import java.time.format.DateTimeFormatter;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.annotation.JsonGetter;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.annotation.JsonSetter;

/**
 * Информация о дополнительном соглашении в результате метода GetPartnerContracts
 * <p>
 * В текущей реализации только часть полей, потому что на момент написания далеко не все из них нам (Директу) нужны;
 * примерный формат результата со всеми данными, которые есть, можно поискать или посмотреть на вики здесь:
 *
 * @link https://wiki.yandex-team.ru/partner/w/external-services-api-balance-get-partner-contracts/
 */
@ParametersAreNonnullByDefault
@JsonIgnoreProperties(ignoreUnknown = true)
public class PartnerContractCollateralInfo {
    private static final DateTimeFormatter DATE_FORMATTER = DateTimeFormatter.ISO_LOCAL_DATE;

    private Long id;

    @JsonProperty("contract2_id")
    private Long contractId2;

    @JsonProperty("collateral_type_id")
    private Long collateralTypeId;

    @JsonProperty("dt")
    private LocalDate startDate;

    @JsonProperty("finish_dt")
    @Nullable
    private LocalDate finishDate;

    @JsonProperty("is_signed")
    @Nullable
    private LocalDate dateOfSigning;

    @JsonProperty("is_faxed")
    @Nullable
    private LocalDate dateOfSigningByFax;

    @JsonProperty("is_cancelled")
    @Nullable
    private LocalDate dateOfCancellation;

    @JsonProperty("is_suspended")
    @Nullable
    private LocalDate dateOfSuspension;

    @JsonProperty("num")
    private String externalCollateralId;

    @JsonProperty("personal_deal_base_pct")
    private BigDecimal personalDealBasePercent;

    public Long getId() {
        return id;
    }

    public void setId(Long id) {
        this.id = id;
    }

    public Long getContractId2() {
        return contractId2;
    }

    public void setContractId2(Long contractId2) {
        this.contractId2 = contractId2;
    }

    public Long getCollateralTypeId() {
        return collateralTypeId;
    }

    public void setCollateralTypeId(Long collateralTypeId) {
        this.collateralTypeId = collateralTypeId;
    }

    public LocalDate getStartDate() {
        return startDate;
    }

    @JsonGetter("dt")
    public String getFormattedStartDate() {
        return DATE_FORMATTER.format(startDate);
    }

    @JsonSetter("dt")
    public void setStartDate(String startDate) {
        setStartDate(DATE_FORMATTER.parse(startDate, LocalDate::from));
    }

    public void setStartDate(LocalDate startDate) {
        this.startDate = startDate;
    }

    @JsonGetter("finish_dt")
    public String getFormattedFinishDate() {
        if (finishDate == null) {
            return "";
        }

        return DATE_FORMATTER.format(finishDate);
    }

    public LocalDate getFinishDate() {
        return finishDate;
    }

    @JsonSetter("finish_dt")
    public void setFinishDate(String finishDate) {
        if (finishDate.isEmpty()) {
            setFinishDate((LocalDate) null);
            return;
        }

        setFinishDate(DATE_FORMATTER.parse(finishDate, LocalDate::from));
    }

    public void setFinishDate(@Nullable LocalDate finishDate) {
        this.finishDate = finishDate;
    }

    @JsonGetter("is_signed")
    public String getFormattedDateOfSigning() {
        if (dateOfSigning == null) {
            return "";
        }

        return DATE_FORMATTER.format(dateOfSigning);
    }

    public LocalDate getDateOfSigning() {
        return dateOfSigning;
    }

    @JsonSetter("is_signed")
    public void setDateOfSigning(String dateOfSigning) {
        if (dateOfSigning.isEmpty()) {
            setDateOfSigning((LocalDate) null);
            return;
        }

        setDateOfSigning(DATE_FORMATTER.parse(dateOfSigning, LocalDate::from));
    }

    public void setDateOfSigning(@Nullable LocalDate dateOfSigning) {
        this.dateOfSigning = dateOfSigning;
    }

    @JsonGetter("is_faxed")
    public String getFormattedDateOfSigningByFax() {
        if (dateOfSigningByFax == null) {
            return "";
        }

        return DATE_FORMATTER.format(dateOfSigningByFax);
    }

    public LocalDate getDateOfSigningByFax() {
        return dateOfSigningByFax;
    }

    @JsonSetter("is_faxed")
    public void setDateOfSigningByFax(String dateOfSigningByFax) {
        if (dateOfSigningByFax.isEmpty()) {
            setDateOfSigningByFax((LocalDate) null);
            return;
        }

        setDateOfSigningByFax(DATE_FORMATTER.parse(dateOfSigningByFax, LocalDate::from));
    }

    public void setDateOfSigningByFax(@Nullable LocalDate dateOfSigningByFax) {
        this.dateOfSigningByFax = dateOfSigningByFax;
    }

    @JsonGetter("is_cancelled")
    public String getFormattedDateOfCancellation() {
        if (dateOfCancellation == null) {
            return "";
        }

        return DATE_FORMATTER.format(dateOfCancellation);
    }

    public LocalDate getDateOfCancellation() {
        return dateOfCancellation;
    }

    @JsonSetter("is_cancelled")
    public void setDateOfCancellation(String dateOfCancellation) {
        if (dateOfCancellation.isEmpty()) {
            setDateOfCancellation((LocalDate) null);
            return;
        }

        setDateOfCancellation(DATE_FORMATTER.parse(dateOfCancellation, LocalDate::from));
    }

    public void setDateOfCancellation(@Nullable LocalDate dateOfCancellation) {
        this.dateOfCancellation = dateOfCancellation;
    }

    @JsonGetter("is_suspended")
    public String getFormattedDateOfSuspension() {
        if (dateOfSuspension == null) {
            return "";
        }

        return DATE_FORMATTER.format(dateOfSuspension);
    }

    public LocalDate getDateOfSuspension() {
        return dateOfSuspension;
    }

    @JsonSetter("is_suspended")
    public void setDateOfSuspension(String dateOfSuspension) {
        if (dateOfSuspension.isEmpty()) {
            setDateOfSuspension((LocalDate) null);
            return;
        }

        setDateOfSuspension(DATE_FORMATTER.parse(dateOfSuspension, LocalDate::from));
    }

    public void setDateOfSuspension(@Nullable LocalDate dateOfSuspension) {
        this.dateOfSuspension = dateOfSuspension;
    }

    public String getExternalCollateralId() {
        return externalCollateralId;
    }

    public void setExternalCollateralId(String externalCollateralId) {
        this.externalCollateralId = externalCollateralId;
    }

    public BigDecimal getPersonalDealBasePercent() {
        return personalDealBasePercent;
    }

    public void setPersonalDealBasePercent(String personalDealBasePercent) {
        this.personalDealBasePercent = new BigDecimal(personalDealBasePercent);
    }

    @Override
    public String toString() {
        return "PartnerContractCollateralInfo{" +
                "id=" + id +
                ", contractId2=" + contractId2 +
                ", collateralTypeId=" + collateralTypeId +
                ", startDate=" + startDate +
                ", dateOfSigning=" + dateOfSigning +
                ", dateOfSigningByFax=" + dateOfSigningByFax +
                ", dateOfCancellation=" + dateOfCancellation +
                ", dateOfSuspension=" + dateOfSuspension +
                ", externalCollateralId='" + externalCollateralId + '\'' +
                ", personalDealBasePercent=" + personalDealBasePercent +
                '}';
    }
}
