package ru.yandex.direct.balance.client.model.response;

import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.util.Map;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.annotation.JsonGetter;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.annotation.JsonSetter;

import static java.util.function.Function.identity;
import static java.util.stream.Collectors.toMap;

/**
 * Информация о договоре в результате метода GetPartnerContracts
 * <p>
 * В текущей реализации только часть полей, потому что на момент написания далеко не все из них нам (Директу) нужны;
 * примерный формат результата со всеми данными, которые есть, можно поискать или посмотреть на вики здесь:
 *
 * @link https://wiki.yandex-team.ru/partner/w/external-services-api-balance-get-partner-contracts/
 */
@ParametersAreNonnullByDefault
@JsonIgnoreProperties(ignoreUnknown = true)
public class PartnerContractContractInfo {
    private static final DateTimeFormatter DATE_FORMATTER = DateTimeFormatter.ISO_LOCAL_DATE;

    @JsonProperty("id")
    private Long id;

    @JsonProperty("contract2_id")
    private Long id2;

    @JsonProperty("dt")
    private LocalDate startDate;

    @JsonProperty("finish_dt")
    @Nullable
    private LocalDate finishDate;

    @JsonProperty("is_signed")
    @Nullable
    private LocalDate dateOfSigning;

    @JsonProperty("is_faxed")
    @Nullable
    private LocalDate dateOfSigningByFax;

    @JsonProperty("is_cancelled")
    @Nullable
    private LocalDate dateOfCancellation;

    @JsonProperty("is_suspended")
    @Nullable
    private LocalDate dateOfSuspension;

    @JsonProperty("services")
    private Set<Long> services;

    @JsonProperty("external_id")
    private String externalContractId;

    @JsonProperty("currency")
    private Long currency;

    public Long getId() {
        return id;
    }

    public void setId(Long id) {
        this.id = id;
    }

    public Long getId2() {
        return id2;
    }

    public void setId2(Long id2) {
        this.id2 = id2;
    }

    public LocalDate getStartDate() {
        return startDate;
    }

    @JsonGetter("dt")
    public String getFormattedStartDate() {
        return DATE_FORMATTER.format(startDate);
    }

    @JsonSetter("dt")
    public void setStartDate(String startDate) {
        setStartDate(DATE_FORMATTER.parse(startDate, LocalDate::from));
    }

    public void setStartDate(LocalDate startDate) {
        this.startDate = startDate;
    }

    @JsonGetter("finish_dt")
    public String getFormattedFinishDate() {
        if (finishDate == null) {
            return "";
        }

        return DATE_FORMATTER.format(finishDate);
    }

    public LocalDate getFinishDate() {
        return finishDate;
    }

    @JsonSetter("finish_dt")
    public void setFinishDate(String finishDate) {
        if (finishDate.isEmpty()) {
            setFinishDate((LocalDate) null);
            return;
        }

        setFinishDate(DATE_FORMATTER.parse(finishDate, LocalDate::from));
    }

    public void setFinishDate(@Nullable LocalDate finishDate) {
        this.finishDate = finishDate;
    }

    @JsonGetter("is_signed")
    public String getFormattedDateOfSigning() {
        if (dateOfSigning == null) {
            return "";
        }

        return DATE_FORMATTER.format(dateOfSigning);
    }

    public LocalDate getDateOfSigning() {
        return dateOfSigning;
    }

    @JsonSetter("is_signed")
    public void setDateOfSigning(String dateOfSigning) {
        if (dateOfSigning.isEmpty()) {
            setDateOfSigning((LocalDate) null);
            return;
        }

        setDateOfSigning(DATE_FORMATTER.parse(dateOfSigning, LocalDate::from));
    }

    public void setDateOfSigning(@Nullable LocalDate dateOfSigning) {
        this.dateOfSigning = dateOfSigning;
    }

    @JsonGetter("is_faxed")
    public String getFormattedDateOfSigningByFax() {
        if (dateOfSigningByFax == null) {
            return "";
        }

        return DATE_FORMATTER.format(dateOfSigningByFax);
    }

    public LocalDate getDateOfSigningByFax() {
        return dateOfSigningByFax;
    }

    @JsonSetter("is_faxed")
    public void setDateOfSigningByFax(String dateOfSigningByFax) {
        if (dateOfSigningByFax.isEmpty()) {
            setDateOfSigningByFax((LocalDate) null);
            return;
        }

        setDateOfSigningByFax(DATE_FORMATTER.parse(dateOfSigningByFax, LocalDate::from));
    }

    public void setDateOfSigningByFax(@Nullable LocalDate dateOfSigningByFax) {
        this.dateOfSigningByFax = dateOfSigningByFax;
    }

    @JsonGetter("is_cancelled")
    public String getFormattedDateOfCancellation() {
        if (dateOfCancellation == null) {
            return "";
        }

        return DATE_FORMATTER.format(dateOfCancellation);
    }

    public LocalDate getDateOfCancellation() {
        return dateOfCancellation;
    }

    @JsonSetter("is_cancelled")
    public void setDateOfCancellation(String dateOfCancellation) {
        if (dateOfCancellation.isEmpty()) {
            setDateOfCancellation((LocalDate) null);
            return;
        }

        setDateOfCancellation(DATE_FORMATTER.parse(dateOfCancellation, LocalDate::from));
    }

    public void setDateOfCancellation(@Nullable LocalDate dateOfCancellation) {
        this.dateOfCancellation = dateOfCancellation;
    }

    @JsonGetter("is_suspended")
    public String getFormattedDateOfSuspension() {
        if (dateOfSuspension == null) {
            return "";
        }

        return DATE_FORMATTER.format(dateOfSuspension);
    }

    public LocalDate getDateOfSuspension() {
        return dateOfSuspension;
    }

    @JsonSetter("is_suspended")
    public void setDateOfSuspension(String dateOfSuspension) {
        if (dateOfSuspension.isEmpty()) {
            setDateOfSuspension((LocalDate) null);
            return;
        }

        setDateOfSuspension(DATE_FORMATTER.parse(dateOfSuspension, LocalDate::from));
    }

    public void setDateOfSuspension(@Nullable LocalDate dateOfSuspension) {
        this.dateOfSuspension = dateOfSuspension;
    }

    public String getExternalContractId() {
        return externalContractId;
    }

    public void setExternalContractId(String externalContractId) {
        this.externalContractId = externalContractId;
    }

    public Set<Long> getServices() {
        return services;
    }

    @JsonGetter("services")
    public Map<Long, Long> getServicesAsMap() {
        return services.stream().collect(toMap(identity(), service -> 1L));
    }

    public void setServices(Set<Long> services) {
        this.services = services;
    }

    @JsonSetter("services")
    public void setServices(Map<Long, Long> services) {
        this.services = services.keySet();
    }

    public Long getCurrency() {
        return currency;
    }

    public void setCurrency(@Nullable Long currency) {
        this.currency = currency;
    }

    @Override
    public String toString() {
        return "PartnerContractContractInfo{" +
                "id=" + id +
                ", id2=" + id2 +
                ", startDate=" + startDate +
                ", finishDate=" + finishDate +
                ", dateOfSigning=" + dateOfSigning +
                ", dateOfSigningByFax=" + dateOfSigningByFax +
                ", dateOfCancellation=" + dateOfCancellation +
                ", dateOfSuspension=" + dateOfSuspension +
                ", services=" + services +
                ", externalContractId='" + externalContractId + '\'' +
                ", currency=" + currency +
                '}';
    }
}
