package ru.yandex.direct.bannerstorage.client;

import java.nio.charset.StandardCharsets;
import java.util.Collection;
import java.util.List;
import java.util.Locale;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import one.util.streamex.StreamEx;
import org.asynchttpclient.AsyncHttpClient;
import org.asynchttpclient.request.body.multipart.ByteArrayPart;
import org.asynchttpclient.request.body.multipart.FilePart;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.asynchttp.ErrorResponseWrapperException;
import ru.yandex.direct.asynchttp.FetcherSettings;
import ru.yandex.direct.asynchttp.ParallelFetcherFactory;
import ru.yandex.direct.asynchttp.Result;
import ru.yandex.direct.bannerstorage.client.model.Creative;
import ru.yandex.direct.bannerstorage.client.model.CreativeGroup;
import ru.yandex.direct.bannerstorage.client.model.CreativeInclude;
import ru.yandex.direct.bannerstorage.client.model.CreativesList;
import ru.yandex.direct.bannerstorage.client.model.ErrorResponse;
import ru.yandex.direct.bannerstorage.client.model.File;
import ru.yandex.direct.bannerstorage.client.model.ItemsRequest;
import ru.yandex.direct.bannerstorage.client.model.Parameter;
import ru.yandex.direct.bannerstorage.client.model.Preview;
import ru.yandex.direct.bannerstorage.client.model.PreviewRequest;
import ru.yandex.direct.bannerstorage.client.model.PreviewsList;
import ru.yandex.direct.bannerstorage.client.model.Template;
import ru.yandex.direct.bannerstorage.client.model.TemplateInclude;
import ru.yandex.direct.bannerstorage.client.model.TemplatesList;
import ru.yandex.direct.bannerstorage.client.model.UploadedFile;
import ru.yandex.direct.http.smart.core.Smart;

import static java.util.Collections.emptyList;
import static ru.yandex.direct.http.smart.error.ErrorUtils.checkResultForErrors;
import static ru.yandex.direct.utils.JsonUtils.fromJson;
import static ru.yandex.direct.utils.JsonUtils.toJson;

/**
 * Клиент к публичному rest api bannerstorage
 * <p>
 * Описание <a href="https://yandex.ru/dev/rtb/doc/api/index-docpage/">Документация</a>}
 */
public class RealBannerStorageClient implements BannerStorageClient {
    private static final Logger logger = LoggerFactory.getLogger(RealBannerStorageClient.class);

    private Api api;
    private UploadApi uploadApi;

    public RealBannerStorageClient(BannerStorageClientConfiguration config, AsyncHttpClient asyncHttpClient) {
        api = createApi(config, asyncHttpClient);
        uploadApi = createUploadApi(config, asyncHttpClient);
    }

    private Api createApi(BannerStorageClientConfiguration config, AsyncHttpClient asyncHttpClient) {
        return Smart.builder()
                .withParallelFetcherFactory(new ParallelFetcherFactory(asyncHttpClient, new FetcherSettings()
                        .withRequestTimeout(config.getTimeout())))
                .addHeaderConfigurator(headers -> {
                    headers.add("Content-type", "application/json");
                    headers.add("Authorization", "OAuth " + config.getToken());
                })
                .withProfileName("banner_storage.client")
                .withBaseUrl(config.getUrl())
                .build()
                .create(Api.class);
    }

    private UploadApi createUploadApi(BannerStorageClientConfiguration config, AsyncHttpClient asyncHttpClient) {
        return Smart.builder()
                .withParallelFetcherFactory(new ParallelFetcherFactory(asyncHttpClient, new FetcherSettings()
                        .withRequestTimeout(config.getFileUploadTimeout())))
                .addHeaderConfigurator(headers -> {
                    headers.add("Authorization", "OAuth " + config.getToken());
                })
                .withProfileName("banner_storage.client")
                .withBaseUrl(config.getUrl())
                .build()
                .create(UploadApi.class);
    }

    private String localeToAcceptLanguage(Locale locale) {
        // в BannerStorage поддержаны в основном две локали: русская и английская
        if (locale.equals(Locale.forLanguageTag("ru"))) {
            return "ru";
        }
        return "en";
    }

    @Override
    public Creative createCreative(Creative creative, Locale locale, CreativeInclude... includes) {
        String includesArg = Stream.of(includes)
                .distinct()
                .map(CreativeInclude::getBannerstorageInclude)
                .collect(Collectors.joining(","));
        String acceptLanguage = localeToAcceptLanguage(locale);
        Result<Creative> result = api.postCreative(creative, includesArg, acceptLanguage).execute();
        checkResultForBannerStorageErrors(result);
        return result.getSuccess();
    }

    @Override
    public Creative updateCreative(Creative creative, Locale locale, CreativeInclude... includes) {
        String includesArg = Stream.of(includes)
                .distinct()
                .map(CreativeInclude::getBannerstorageInclude)
                .collect(Collectors.joining(","));
        String acceptLanguage = localeToAcceptLanguage(locale);
        Result<Creative> result = api.putCreative(creative.getId(), creative, includesArg, acceptLanguage).execute();
        checkResultForBannerStorageErrors(result);
        return result.getSuccess();
    }

    private <T> void checkResultForBannerStorageErrors(Result<T> result) {
        if (result.getErrors() != null && !result.getErrors().isEmpty()) {
            Throwable throwable = result.getErrors().get(0);
            if (throwable instanceof ErrorResponseWrapperException) {
                ErrorResponseWrapperException exception = (ErrorResponseWrapperException) throwable;
                if (exception.getResponse() != null) {
                    if (exception.getResponse().getStatusCode() == 422) {
                        byte[] responseBody = exception.getResponse().getResponseBodyAsBytes();
                        ErrorResponse errorResponse = fromJson(responseBody, ErrorResponse.class);
                        throw new BsCreativeValidationFailedException(errorResponse);
                    } else if (exception.getResponse().getStatusCode() == 404) {
                        throw new BsCreativeNotFoundException();
                    }
                }
            }
        }
        checkResultForErrors(result, BannerStorageClientException::new);
    }

    public ItemsRequest addTnsArticles(Integer creativeId, ItemsRequest request) {
        Result<ItemsRequest> result = api.putTnsArticles(creativeId, request).execute();
        checkResultForErrors(result, BannerStorageClientException::new, creativeId, request);
        return result.getSuccess();
    }

    public ItemsRequest addTnsBrands(Integer creativeId, ItemsRequest request) {
        Result<ItemsRequest> result = api.putTnsBrands(creativeId, request).execute();
        checkResultForErrors(result, BannerStorageClientException::new, creativeId, request);
        return result.getSuccess();
    }

    public Creative requestModeration(Integer creativeId) {
        logger.info("requestModeration request for id: {}", creativeId);
        Result<Creative> result = api.requestModeration(creativeId).execute();
        checkResultForErrors(result, BannerStorageClientException::new, creativeId);
        logger.info("requestModeration result: {}", result);
        return result.getSuccess();
    }

    @Override
    public Creative requestEdit(Integer creativeId) {
        logger.info("requestEdit request for id: {}", creativeId);
        Result<Creative> result = api.requestEdit(creativeId).execute();
        checkResultForErrors(result, BannerStorageClientException::new, creativeId);
        logger.info("requestEdit result: {}", result);
        return result.getSuccess();
    }

    @Override
    public List<Creative> getCreatives(Collection<Integer> creativeIds, CreativeInclude... includes) {
        if (creativeIds.isEmpty()) {
            return emptyList();
        }
        if (creativeIds.size() > MAX_CREATIVES_IN_BATCH) {
            throw new IllegalArgumentException(
                    String.format("creativeIds limit exceeded: %d > %d", creativeIds.size(), MAX_CREATIVES_IN_BATCH));
        }
        String includesArg = Stream.of(includes)
                .distinct()
                .map(CreativeInclude::getBannerstorageInclude)
                .collect(Collectors.joining(","));
        String creativeIdsArg = creativeIds.stream()
                .distinct()
                .map(Object::toString)
                .collect(Collectors.joining(","));
        logger.info("getCreatives request for ids: {} with includes: {}", creativeIdsArg, includesArg);
        Result<CreativesList> result = api.getCreatives(creativeIdsArg, includesArg).execute();
        checkResultForBannerStorageErrors(result);
        return result.getSuccess().getItems();
    }

    @Override
    public Creative setModeratedExternally(Integer creativeId) {
        logger.info("setModeratedExternally request for id: {}", creativeId);
        Result<Creative> result = api.setModeratedExternally(creativeId).execute();
        checkResultForErrors(result, BannerStorageClientException::new, creativeId);
        logger.info("setModeratedExternally result: {}", toJson(result));
        return result.getSuccess();
    }

    @Override
    public File getFile(Integer fileId) {
        logger.info("getFile request for id: {}", fileId);
        Result<File> result = api.getFile(fileId).execute();
        checkResultForErrors(result, BannerStorageClientException::new, fileId);
        logger.info("getFile result: {}", toJson(result));
        return result.getSuccess();
    }

    @Override
    public List<Template> getTemplates(TemplateInclude... includes) {
        String includesArg = Stream.of(includes)
                .distinct()
                .map(TemplateInclude::getApiConstant)
                .collect(Collectors.joining(","));
        Result<TemplatesList> result = api.getTemplates(includesArg).execute();
        checkResultForErrors(result, BannerStorageClientException::new, includesArg);
        return result.getSuccess().getItems();
    }

    @Override
    public Template getTemplate(int templateId, TemplateInclude... includes) {
        String includesArg = Stream.of(includes)
                .distinct()
                .map(TemplateInclude::getApiConstant)
                .collect(Collectors.joining(","));
        Result<Template> result = api.getTemplate(templateId, includesArg).execute();
        checkResultForErrors(result, BannerStorageClientException::new, templateId, includesArg);
        logger.info("getTemplate result: {}", toJson(result));
        return result.getSuccess();
    }

    @Override
    public void approveExternallyModeratedCreative(Integer creativeId, Integer creativeVersionId) {
        logger.info("approveExternallyModeratedCreative request for creative {} version {}", creativeId,
                creativeVersionId);
        Result<Creative> result = api.approveExternallyModeratedCreative(creativeId, creativeVersionId).execute();
        checkResultForErrors(result, BannerStorageClientException::new, creativeId, creativeVersionId);
        logger.info("approveExternallyModeratedCreative result: {}", toJson(result));
    }

    @Override
    public void rejectExternallyModeratedCreative(Integer creativeId, Integer creativeVersionId) {
        logger.info("rejectExternallyModeratedCreative request for creative {} version {}", creativeId,
                creativeVersionId);
        Result<Creative> result = api.rejectExternallyModeratedCreative(creativeId, creativeVersionId).execute();
        checkResultForErrors(result, BannerStorageClientException::new, creativeId, creativeVersionId);
        logger.info("rejectExternallyModeratedCreative result: {}", toJson(result));
    }

    @Override
    public UploadedFile uploadFile(byte[] data, String fileName) {
        if (data.length > BannerStorageClient.MAX_FILE_SIZE) {
            throw new RuntimeException("BannerStorage MAX_FILE_SIZE limit exceeded");
        }
        logger.info("uploadFile request for file {}", fileName);
        Result<UploadedFile> result = uploadApi.uploadFile(
                new ByteArrayPart("file", data, "multipart/form-data", StandardCharsets.UTF_8, fileName)).execute();
        checkResultForErrors(result, BannerStorageClientException::new, fileName);
        logger.info("uploadFile result: {}", result);
        return result.getSuccess();
    }

    @Override
    public UploadedFile uploadFile(java.io.File file, String origFileName) {
        if (file.length() > BannerStorageClient.MAX_FILE_SIZE) {
            throw new RuntimeException("BannerStorage MAX_FILE_SIZE limit exceeded");
        }
        logger.info("uploadFile request for file {}", origFileName);
        Result<UploadedFile> result = uploadApi.uploadFile(
                new FilePart("file", file, "multipart/form-data", StandardCharsets.UTF_8, origFileName)).execute();
        checkResultForErrors(result, BannerStorageClientException::new, origFileName);
        logger.info("uploadFile result: {}", result);
        return result.getSuccess();
    }

    @Override
    public CreativeGroup getSmartCreativeGroup(int id, CreativeInclude... includes) {
        logger.info("getSmartCreativeGroup request for id {}", id);
        String includesArg = StreamEx.of(includes)
                .distinct()
                .map(CreativeInclude::getBannerstorageInclude)
                .map(include -> "creatives." + include)
                .prepend("creatives")
                .joining(",");
        Result<CreativeGroup> result = api.getCreativeGroup(id, includesArg).execute();
        checkResultForErrors(result, BannerStorageClientException::new, id);
        logger.info("getSmartCreativeGroup result: {}", toJson(result));
        return result.getSuccess();
    }

    @Override
    public List<Preview> getSmartGroupPreviews(List<Integer> creativeIds, List<Parameter> modifiedParameters,
                                               String dcParams) {
        PreviewRequest previewRequest = new PreviewRequest(creativeIds, null, modifiedParameters, dcParams);
        Result<PreviewsList> result = api.getPreviews(previewRequest).execute();
        checkResultForErrors(result, BannerStorageClientException::new, previewRequest);
        return result.getSuccess().getItems();
    }

    @Override
    public List<Preview> getSmartGroupPreviews(int templateId, List<Parameter> modifiedParameters, String dcParams) {
        PreviewRequest previewRequest = new PreviewRequest(null, templateId, modifiedParameters, dcParams);
        Result<PreviewsList> result = api.getPreviews(previewRequest).execute();
        checkResultForErrors(result, BannerStorageClientException::new, previewRequest);
        return result.getSuccess().getItems();
    }

    @Override
    public CreativeGroup createSmartCreativeGroup(CreativeGroup creativeGroup) {
        logger.info("createSmartCreativeGroup request: {}", toJson(creativeGroup));
        Result<CreativeGroup> result = api.createCreativeGroup(
                creativeGroup,
                true,
                "businessType,layoutCode,previewUrl,isPredeployed"
        ).execute();
        checkResultForErrors(result, BannerStorageClientException::new, creativeGroup);
        logger.info("createSmartCreativeGroup result: {}", toJson(result));
        return result.getSuccess();
    }

    @Override
    public CreativeGroup editSmartCreativeGroup(int id, CreativeGroup creativeGroup) {
        logger.info("editSmartCreativeGroup request: {}", toJson(creativeGroup));
        Result<CreativeGroup> result = api.editCreativeGroup(
                id,
                creativeGroup,
                true,
                "businessType,layoutCode,previewUrl,isPredeployed"
        ).execute();
        checkResultForErrors(result, BannerStorageClientException::new, id, creativeGroup);
        logger.info("editSmartCreativeGroup result: {}", toJson(result));
        return result.getSuccess();
    }
}
