package ru.yandex.direct.bannersystem;

import java.net.URI;
import java.net.URISyntaxException;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.UUID;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import io.netty.handler.codec.http.DefaultHttpHeaders;
import io.netty.handler.codec.http.HttpHeaderNames;
import io.netty.handler.codec.http.HttpHeaders;
import io.netty.handler.codec.http.HttpMethod;
import org.apache.http.NameValuePair;
import org.apache.http.client.utils.URIBuilder;
import org.asynchttpclient.AsyncHttpClient;
import org.asynchttpclient.RequestBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.asynchttp.FetcherSettings;
import ru.yandex.direct.asynchttp.ParallelFetcher;
import ru.yandex.direct.asynchttp.ParallelFetcherFactory;
import ru.yandex.direct.asynchttp.ParsableStringRequest;
import ru.yandex.direct.asynchttp.Result;
import ru.yandex.direct.bannersystem.exception.BsClientException;
import ru.yandex.direct.bannersystem.handle.BsHandleSpec;
import ru.yandex.direct.solomon.SolomonUtils;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.tracing.TraceProfile;
import ru.yandex.direct.utils.SystemUtils;

import static ru.yandex.direct.utils.CommonUtils.nvl;

/**
 * Базовый клиент для ручек БК. Умеет делать обращения к различным ручкам ({@link BsHandleSpec})
 * и преобразовывать их ответ в объекты.
 * <p>
 * Клиент не содержит в себе логики, специфичной для какой-либо ручки, такую логику нужно выносить в отдельные клиенты
 * для этих ручек.
 */
@ParametersAreNonnullByDefault
public class BannerSystemClient {
    private static final Logger logger = LoggerFactory.getLogger(BannerSystemClient.class);

    static final String REQUEST_UUID_HEADER_NAME = "Request-uuid";
    private static final String BS_CLIENT_TRACE = "bsclient";
    public static final Duration BANNER_SYSTEM_CLIENT_REQUEST_TIMEOUT = Duration.ofSeconds(60);

    private final BsUriFactory bsUriFactory;
    private final ParallelFetcherFactory parallelFetcherFactory;

    public BannerSystemClient(BsUriFactory bsUriFactory, AsyncHttpClient asyncHttpClient) {
        this.bsUriFactory = bsUriFactory;

        FetcherSettings settings = new FetcherSettings()
                .withRequestTimeout(BANNER_SYSTEM_CLIENT_REQUEST_TIMEOUT);
        parallelFetcherFactory = new ParallelFetcherFactory(asyncHttpClient, settings);
    }

    public BannerSystemClient(BsUriFactory bsUriFactory, ParallelFetcherFactory parallelFetcherFactory) {
        this.bsUriFactory = bsUriFactory;
        this.parallelFetcherFactory = parallelFetcherFactory;
    }

    /**
     * Получить полный путь к продовой ручке для заданного контроллера с указанными параметрами GET-запроса
     */
    private String buildUri(BsHandleSpec bsHandle, List<NameValuePair> parameters) {
        URI uri;
        try {
            URI baseUri = bsUriFactory.getProdUri(bsHandle);
            if (parameters.isEmpty()) {
                uri = baseUri;
            } else {
                URIBuilder builder = new URIBuilder(baseUri);
                builder.addParameters(parameters);
                uri = builder.build();
            }
        } catch (URISyntaxException e) {
            throw new BsClientException("Could not build URI for request", e);
        }
        return uri.toASCIIString();
    }

    /**
     * Сделать запрос к ручке БК с заданным таимаутом и вернуть ее ответ в виде строки.
     */
    <T, P> P doRequest(BsHandleSpec<T, P> bsHandle, List<NameValuePair> parameters, Duration timeout) {
        return doRequest(bsHandle, null, parameters, UUID.randomUUID(), timeout);
    }

    /**
     * Сделать запрос к ручке БК с заданным таимаутом и вернуть ее ответ в виде строки.
     */
    public <T, P> P doRequest(BsHandleSpec<T, P> bsHandle, T body, UUID requestUuid, Duration timeout) {
        return doRequest(bsHandle, body, Collections.emptyList(), requestUuid, timeout);
    }

    /**
     * Сделать запрос к ручке БК с заданным таимаутом и вернуть ее ответ в виде строки.
     */
    <T, P> P doRequest(BsHandleSpec<T, P> bsHandle, @Nullable T body, List<NameValuePair> parameters,
                       UUID requestUuid, Duration timeout) {
        String bodyString = body != null ? serializeRequestBody(bsHandle, body, requestUuid) : null;

        String uriString = buildUri(bsHandle, parameters);
        HttpHeaders requestHeaders = new DefaultHttpHeaders()
                .add(HttpHeaderNames.CONTENT_TYPE, bsHandle.getRequestContentType().toString())
                .add(REQUEST_UUID_HEADER_NAME, requestUuid.toString())
                .add(bsHandle.getAdditionalHeaders());

        String responseString;
        try (TraceProfile profile = Trace.current().profile(traceName(bsHandle, "do_request"))) {
            logRequestOrResponseData(uriString, bodyString, requestUuid, "request", timeout);
            responseString = doRawRequest(uriString, bodyString, requestHeaders, timeout, profile.getFunc());
            logRequestOrResponseData(uriString, responseString, requestUuid, "response", timeout);
        }

        return deserializeResponseBody(bsHandle, responseString, requestUuid);
    }

    /**
     * Сделать запрос к ручке БК с заданным таимаутом и вернуть ее ответ в виде строки.
     */
    String doRawRequest(String uriString, @Nullable String body, HttpHeaders headers, Duration timeout, String label) {
        FetcherSettings fetcherSettings = parallelFetcherFactory.defaultSettingsCopy()
                .withRequestTimeout(timeout)
                .withMetricRegistry(SolomonUtils.getParallelFetcherMetricRegistry(label));
        try (ParallelFetcher<String> fetcher = parallelFetcherFactory.getParallelFetcher(fetcherSettings)) {
            RequestBuilder builder = new RequestBuilder()
                    .setUrl(uriString)
                    .setHeaders(headers);
            if (body != null) {
                builder.setBody(body);
                builder.setMethod(HttpMethod.POST.name());
            }

            Result<String> result;
            try {
                result = fetcher.execute(new ParsableStringRequest(0, builder.build()));
            } catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                throw new BsClientException("Got error while performing query", e);
            }

            if (result.getErrors() != null && !result.getErrors().isEmpty()) {
                BsClientException exception = new BsClientException("Got error HTTP error in response");
                result.getErrors().forEach(exception::addSuppressed);
                throw exception;
            }

            return result.getSuccess();
        }
    }

    private <T> String serializeRequestBody(BsHandleSpec<T, ?> bsHandle, T bodyObj, UUID requestUuid) {
        try (TraceProfile ignore = Trace.current().profile(traceName(bsHandle, "serialize_request"))) {
            return bsHandle.serializeRequestBody(bodyObj);
        } catch (Exception e) {
            logErrorData(e.getMessage(), requestUuid, "serialization_error");
            throw new BsClientException("Got error while serializing query", e);
        }
    }

    private <T> T deserializeResponseBody(BsHandleSpec<?, T> bsHandle, String responseBody, UUID requestUuid) {
        try (TraceProfile ignore = Trace.current().profile(traceName(bsHandle, "deserialize_response"))) {
            return bsHandle.deserializeResponseBody(responseBody);
        } catch (Exception e) {
            logErrorData(e.getMessage(), requestUuid, "deserialization_error");
            throw new BsClientException("Got error while deserializing response", e);
        }
    }

    private String traceName(BsHandleSpec<?, ?> bsHandle, String... additions) {
        List<String> lines = new ArrayList<>();
        lines.addAll(Arrays.asList(BS_CLIENT_TRACE, bsHandle.getHandleName()));
        lines.addAll(Arrays.asList(additions));
        return String.join(":", lines);
    }

    private void logRequestOrResponseData(String uriString, @Nullable String body, UUID requestUuid, String dataType,
                                          Duration timeout) {
        if (logger.isInfoEnabled()) {
            String bodyPart = nvl(body, "NO BODY");
            logger.info("[pid={},reqid={},uri={},uuid={},data_type={},timeout={}]\t{}",
                    SystemUtils.getPid(),
                    Trace.current().getSpanId(),
                    uriString,
                    requestUuid,
                    dataType,
                    timeout.toMillis(),
                    bodyPart);
        }
    }

    private void logErrorData(String message, UUID requestUuid, String dataType) {
        if (logger.isErrorEnabled()) {
            logger.error("[pid={},reqid={},uuid={},data_type={}]\t{}",
                    SystemUtils.getPid(),
                    Trace.current().getSpanId(),
                    requestUuid,
                    dataType,
                    message);
        }
    }
}
