package ru.yandex.direct.bannersystem;

import java.time.Duration;
import java.util.Collections;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.databind.JavaType;
import org.apache.http.message.BasicNameValuePair;

import ru.yandex.direct.bannersystem.container.exporttable.BsTableSpec;
import ru.yandex.direct.bannersystem.container.exporttable.HolidayInfoRecord;
import ru.yandex.direct.bannersystem.container.exporttable.SspInfoRecord;
import ru.yandex.direct.bannersystem.exception.BsClientException;
import ru.yandex.direct.bannersystem.exception.BsExportTableException;
import ru.yandex.direct.bannersystem.model.exporttable.UniwordRecord;
import ru.yandex.direct.utils.JsonUtils;

/**
 * Клиент для ручки выгрузки таблиц в БК.
 */
@ParametersAreNonnullByDefault
public class BsExportTableClient {
    public static final BsTableSpec<SspInfoRecord> SSP_INFO = new BsTableSpec<>("SSPInfo", SspInfoRecord.class);
    public static final BsTableSpec<UniwordRecord> UNIWORD = new BsTableSpec<>("Uniword", UniwordRecord.class);
    private static final BsTableSpec<HolidayInfoRecord> HOLIDAY_INFO =
            new BsTableSpec<>("Holiday", HolidayInfoRecord.class);

    private static final Duration DEFAULT_TIMEOUT = Duration.ofSeconds(30);
    private static final String TABLE_NAME_QUERY_PARAM = "table_name";

    private final BannerSystemClient bannerSystemClient;

    public BsExportTableClient(BannerSystemClient bannerSystemClient) {
        this.bannerSystemClient = bannerSystemClient;
    }

    /**
     * Получить из БК таблицу в виде списка объектов, представляющих ее строки.
     */
    private <T> List<T> getBsTableRowsList(BsTableSpec bsTableSpec) {
        String tableName = bsTableSpec.getTableName();
        BasicNameValuePair param = new BasicNameValuePair(TABLE_NAME_QUERY_PARAM, tableName);

        JavaType type = JsonUtils.getObjectMapper().getTypeFactory()
                .constructCollectionType(List.class, bsTableSpec.getRecordClass());
        try {
            return bannerSystemClient
                    .doRequest(BsUriFactory.EXPORT_TABLE.withResponseClass(type), Collections.singletonList(param),
                            DEFAULT_TIMEOUT);
        } catch (BsClientException e) {
            throw new BsExportTableException("Got error while performing query", e);
        }
    }

    /**
     * Получить из БК таблицу SSPInfo ({@link SspInfoRecord})
     */
    public List<SspInfoRecord> getSspInfoTableRowsList() {
        return getBsTableRowsList(SSP_INFO);
    }

    /**
     * Получить из БК таблицу Holiday ({@link HolidayInfoRecord})
     * <p>
     * БК возвращает JSON вида
     * [{"Options":"","VirtualWeekday":"0","WDay":"1","UpdateTime":"2006-03-08 00:00:00","Coeff":"1","RegionID":"225"}]
     */
    public List<HolidayInfoRecord> getHolidayInfoTableRowsList() {
        return getBsTableRowsList(HOLIDAY_INFO);
    }

    /**
     * получить таблицу из БК Uniword ({@link UniwordRecord})
     */
    public List<UniwordRecord> getUniwordTableRows() {
        return getBsTableRowsList(UNIWORD);
    }
}
