package ru.yandex.direct.bannersystem;

import java.net.URI;
import java.net.URISyntaxException;
import java.util.EnumMap;
import java.util.List;
import java.util.Map;

import javax.annotation.ParametersAreNonnullByDefault;

import org.apache.http.client.utils.URIBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.bannersystem.container.appstoredata.BsImportAppStoreDataItem;
import ru.yandex.direct.bannersystem.container.appstoredata.BsImportAppStoreDataResponse;
import ru.yandex.direct.bannersystem.container.masterreport.MasterReportRequest;
import ru.yandex.direct.bannersystem.container.masterreport.MasterReportResponse;
import ru.yandex.direct.bannersystem.exception.BsUriFactoryException;
import ru.yandex.direct.bannersystem.handle.BsHandleSpec;
import ru.yandex.direct.bannersystem.handle.BsJsonHandleSpec;
import ru.yandex.direct.bannersystem.handle.BsUpdateData2Spec;

/**
 * Фабрика, поставляющая URL для обращения к различным ручкам БК по их описанию
 * в формате {@link BsHandleSpec}
 */
@ParametersAreNonnullByDefault
public class BsUriFactory {
    public static final BsJsonHandleSpec<?, ?> EXPORT_TABLE =
            new BsJsonHandleSpec<>("/export/export-table.cgi", BsHostType.EXPORT, false);
    public static final BsJsonHandleSpec<MasterReportRequest, MasterReportResponse> MASTER_REPORT =
            new BsJsonHandleSpec<>("/export/master-report.cgi", BsHostType.FAST_EXPORT, true,
                    MasterReportResponse.class);
    public static final BsJsonHandleSpec<List<BsImportAppStoreDataItem>, BsImportAppStoreDataResponse>
            IMPORT_APPLICATION_STORE_DATA =
            new BsJsonHandleSpec<>("/export/import-application-store-data.cgi", BsHostType.IMPORT, false,
                    BsImportAppStoreDataResponse.class);

    public static final BsUpdateData2Spec UPDATE_DATA_2_SPEC =
            new BsUpdateData2Spec("", BsHostType.IMPORT, true);


    private static final Logger logger = LoggerFactory.getLogger(BsUriFactory.class);

    private final EnumMap<BsHostType, URI> prodUrls = new EnumMap<>(BsHostType.class);
    private final EnumMap<BsHostType, URI> preProdUrls = new EnumMap<>(BsHostType.class);

    /**
     * Инициализировать фабрику, отдающую только production URL
     */
    public BsUriFactory(Map<BsHostType, String> prodUrls) {
        fillUrlsMap(this.prodUrls, prodUrls);
    }

    /**
     * Инициализировать фабрику, отдающую как production так и pre-prod URL
     */
    public BsUriFactory(Map<BsHostType, String> prodUrls, Map<BsHostType, String> preProdUrls) {
        this(prodUrls);
        fillUrlsMap(this.preProdUrls, preProdUrls);
    }

    private void fillUrlsMap(Map<BsHostType, URI> target, Map<BsHostType, String> source) {
        source.forEach((type, uri) -> target.put(type, convertStringToUri(uri)));
    }

    URI convertStringToUri(String source) {
        URI result;
        try {
            result = new URI(source);
        } catch (URISyntaxException e) {
            throw new BsUriFactoryException("Invalid hostname: " + source, e);
        }

        if (result.getScheme() == null || result.getHost() == null) {
            throw new BsUriFactoryException("Invalid hostname: " + source);
        }
        return result;
    }

    /**
     * Получить production URL ручки по ее опичанию
     */
    public URI getProdUri(BsHandleSpec bsHandle) {
        return getUri(bsHandle, false);
    }

    /**
     * Получить pre-prod URL ручки по ее опичанию
     */
    public URI getPreProdUri(BsHandleSpec bsHandle) {
        return getUri(bsHandle, true);
    }

    private URI getUri(BsHandleSpec bsHandle, boolean needPreProd) {
        if (needPreProd && !bsHandle.hasPreProd()) {
            throw new BsUriFactoryException(
                    "Cannot provide preprod URI for handle without preprod: " + bsHandle.getUrlPath());
        }

        URI baseUri = (needPreProd ? preProdUrls : prodUrls).get(bsHandle.getHostType());
        if (baseUri == null) {
            throw new BsUriFactoryException("Cannot provide URI for host type " + bsHandle.getHostType());
        }

        URIBuilder builder = new URIBuilder(baseUri.resolve(bsHandle.getUrlPath()));
        URI resultUri;
        try {
            resultUri = builder.build();
        } catch (URISyntaxException e) {
            throw new BsUriFactoryException(
                    "Building an URI for " + bsHandle.getUrlPath() + " resulted in invalid result", e);
        }

        if (logger.isTraceEnabled()) {
            logger.trace(resultUri.toASCIIString());
        }

        return resultUri;
    }
}
