package ru.yandex.direct.bannersystem.container.masterreport

import com.fasterxml.jackson.annotation.JsonIgnoreProperties
import com.fasterxml.jackson.annotation.JsonProperty
import ru.yandex.direct.bannersystem.container.masterreport.converter.calcDepth
import ru.yandex.direct.bannersystem.container.masterreport.dict.MasterReportCampaignPlatform
import java.math.BigDecimal

const val MULTI_GOALS_DATA = "multiGoalsData"

/**
 * Ответ от БК
 *
 * Предполагается, что поля, начинающиеся с префикса raw, скорее не самостоятельны, а используются
 * для вычисления других полей
 */
@JsonIgnoreProperties(ignoreUnknown = true)
data class MasterReportRow(
        @JsonProperty(BS_PERIOD)
        var period: String? = null,

        @JsonProperty(BS_EVENT_TIME)
        var eventTime: String? = null,

        @JsonProperty(BS_CAMPAIGN_ID)
        var campaignId: Long? = null,

        @JsonProperty(BS_PLATFORM)
        var platform: MasterReportCampaignPlatform? = null,

        @JsonProperty(BS_PAGE_NAME)
        var pageName: String? = null,

        @JsonProperty(BS_SHOWS)
        var shows: Long? = null,

        @JsonProperty(BS_CLICKS)
        var clicks: Long? = null,

        @JsonProperty(BS_CONVERSIONS)
        var conversions: Long? = null,

        @JsonProperty(BS_COST_PER_CONVERSION)
        var costPerConversion: BigDecimal? = null,

        @JsonProperty(BS_AVG_CPC)
        var avgCpc: BigDecimal? = null,

        @JsonProperty(BS_COST)
        var cost: BigDecimal? = null,

        @JsonProperty(BS_CTR)
        var ctr: Double? = null,

        @JsonProperty(BS_CPM)
        var cpm: Double? = null,

        @JsonProperty(BS_RF)
        var rf: Double? = null,

        @JsonProperty(BS_CONVERSION_RATE)
        var conversionRate: Double? = null,

        @JsonProperty(BS_INCOME)
        var income: BigDecimal? = null,

        @JsonProperty(BS_PROFIT)
        var profit: BigDecimal? = null,

        @JsonProperty(BS_UNIQ_VIEWERS)
        var uniqViewers: Long? = null,

        @JsonProperty(BS_ROI)
        var roi: Double? = null,

        @JsonProperty(BS_CRR)
        var crr: Double? = null,

        @JsonProperty(BS_BOUNCE_RATIO)
        var bounceRatio: Double? = null,

        @JsonProperty(BS_BOUNCES)
        var bounces: Long? = null,

        @JsonProperty(BS_SESSION_DEPTH)
        var rawSessionDepth: Long? = null,

        @JsonProperty(BS_SESSION_NUM)
        var rawSessionNum: Long? = null,

        @JsonProperty(BS_SESSION_NUM_LIM)
        var rawSessionNumLim: Long? = null,

        @JsonProperty(MULTI_GOALS_DATA)
        var multiGoalsData: List<MultiGoalsData>? = null
) {
    /**
     * В БК нет глубины просмотра страницы в том виде, в котором она нужна нам. Поэтому считаем на своей стороне
     */
    val depth: Double?
        get() = calcDepth(rawSessionNumLim, rawSessionNum, rawSessionDepth, clicks)

    @JsonIgnoreProperties(ignoreUnknown = true)
    data class MultiGoalsData(
            @JsonProperty(BS_MULTI_GOALS_ATTRIBUTION_TYPE)
            val attributionType: Int,
            @JsonProperty(BS_MULTI_GOALS_ID)
            val id: Long,
            @JsonProperty(BS_MULTI_CONVERSIONS)
            val conversions: Long? = null,
            @JsonProperty(BS_MULTI_GOALS_COST_PER_CONVERSION)
            val costPerConversion: BigDecimal? = null,
            @JsonProperty(BS_MULTI_GOALS_CONVERSION_RATE)
            val conversionRate: Double? = null,
            @JsonProperty(BS_MULTI_GOALS_INCOME)
            val income: BigDecimal? = null,
            @JsonProperty(BS_MULTI_GOALS_ROI)
            val roi: Double? = null,
            @JsonProperty(BS_MULTI_GOALS_CRR)
            val crr: Double? = null,
    ) {
        /**
         * Прибыль
         */
        fun profit(cost: BigDecimal?) = absDelta(income, cost)

        fun conversionsAbsDelta(r: MultiGoalsData?) = absDelta(conversions, r?.conversions)
        fun costPerConversionAbsDelta(r: MultiGoalsData?) = absDelta(costPerConversion, r?.costPerConversion)
        fun conversionRateAbsDelta(r: MultiGoalsData?) = absDelta(conversionRate, r?.conversionRate)
        fun incomeAbsDelta(r: MultiGoalsData?) = absDelta(income, r?.income)
        fun roiAbsDelta(r: MultiGoalsData?) = absDelta(roi, r?.roi)
        fun crrAbsDelta(r: MultiGoalsData?) = absDelta(crr, r?.crr)
    }

    fun showsAbsDelta(r: MasterReportRow?) = absDelta(shows, r?.shows)
    fun clicksAbsDelta(r: MasterReportRow?) = absDelta(clicks, r?.clicks)
    fun conversionsAbsDelta(r: MasterReportRow?) = absDelta(conversions, r?.conversions)
    fun costAbsDelta(r: MasterReportRow?) = absDelta(cost, r?.cost)
    fun incomeAbsDelta(r: MasterReportRow?) = absDelta(income, r?.income)
    fun profitAbsDelta(r: MasterReportRow?) = absDelta(profit, r?.profit)
    fun uniqViewersAbsDelta(r: MasterReportRow?) = absDelta(uniqViewers, r?.uniqViewers)
    fun depthAbsDelta(r: MasterReportRow?) = absDelta(depth, r?.depth)
    fun costPerConversionAbsDelta(r: MasterReportRow?) = absDelta(costPerConversion, r?.costPerConversion)
    fun avgCpcAbsDelta(r: MasterReportRow?) = absDelta(avgCpc, r?.avgCpc)
    fun ctrAbsDelta(r: MasterReportRow?) = absDelta(ctr, r?.ctr)
    fun conversionRateAbsDelta(r: MasterReportRow?) = absDelta(conversionRate, r?.conversionRate)
    fun roiAbsDelta(r: MasterReportRow?) = absDelta(roi, r?.roi)
    fun crrAbsDelta(r: MasterReportRow?) = absDelta(crr, r?.crr)
    fun bounceRatioAbsDelta(r: MasterReportRow?) = absDelta(bounceRatio, r?.bounceRatio)

}

private fun absDelta(first: Long?, second: Long?) =
        if (first == null || second == null) {
            null
        } else {
            first - second
        }

private fun absDelta(first: Double?, second: Double?) =
        if (first == null || second == null) {
            null
        } else {
            first - second
        }

fun absDelta(first: BigDecimal?, second: BigDecimal?) =
        if (first == null || second == null) {
            null
        } else {
            first - second
        }
