package ru.yandex.direct.bannersystem.container.masterreport.converter

import com.fasterxml.jackson.core.JsonParser
import com.fasterxml.jackson.databind.DeserializationContext
import com.fasterxml.jackson.databind.JsonDeserializer
import com.fasterxml.jackson.databind.JsonNode
import com.fasterxml.jackson.databind.annotation.JsonDeserialize
import com.fasterxml.jackson.databind.node.ArrayNode
import com.fasterxml.jackson.databind.node.ObjectNode
import ru.yandex.direct.bannersystem.container.masterreport.MULTI_GOALS_DATA
import ru.yandex.direct.bannersystem.container.masterreport.MasterReportResponse
import ru.yandex.direct.utils.JsonUtils

@JsonDeserialize
// Т.к. нет простого и удобного способа передать управление дефолтному десериализатору в кастомном десериализаторе,
// сделали фиктивный класс, наследуемый от нужного, у которого пустой десериализатор, чтобы остановить
// бесконечный вызов десериализаторов вплоть до StackOverflow
class MasterReportResponseFake : MasterReportResponse()

class MasterReportResponseDeserializer : JsonDeserializer<MasterReportResponseFake>() {
    override fun deserialize(jp: JsonParser, ctxt: DeserializationContext): MasterReportResponseFake {
        val codec = jp.codec
        val json: JsonNode = codec.readTree(jp)
        preprocessData(json)
        return codec.treeToValue(json, MasterReportResponseFake::class.java)
    }

    private fun preprocessData(json: JsonNode): JsonNode {
        val obj = json as ObjectNode

        val header = mutableListOf<String>()
        val multiGoalsHeader = mutableListOf<String>()
        val headerNode = json["header"] as ArrayNode
        for (jsonNode in headerNode) {
            if (jsonNode.isTextual) {
                header.add(jsonNode.asText())
            }
            if (jsonNode.isArray) {
                for (gNode in (jsonNode as ArrayNode)) {
                    multiGoalsHeader.add(gNode.textValue())
                }
            }
        }

        val dataNode = obj["data"] as ArrayNode
        val transformedData = transformData(dataNode, header, multiGoalsHeader)
        obj.set("data", transformedData)

        val totalsNode = obj["totals"]
        val transformedTotals = transformTotals(totalsNode)
        obj.set("totals", transformedTotals)

        return obj
    }

    private fun transformData(
            dataNode: ArrayNode,
            header: List<String>,
            multiGoalsHeader: List<String>
    ): ArrayNode {
        val rows = dataNode.map {
            it.zipWithHeaders(header, multiGoalsHeader)
        }
        return JsonUtils.MAPPER.valueToTree(rows)
    }

    private fun JsonNode.zipWithHeaders(header: List<String>, multiGoalsHeader: List<String>): Map<String, JsonNode> {
        val map = mutableMapOf<String, JsonNode>()
        val metrics = this as ArrayNode
        header.zip(metrics).forEach { map[it.first] = it.second }

        if (multiGoalsHeader.isEmpty()) {
            return map
        }
        val index = findMultiGoalsIndex(metrics)
        if (index == -1) {
            return map
        }
        val mgMetrics = metrics[index] as ArrayNode
        val mgArray = mgMetrics.map { multiGoalsHeader.zip(it as ArrayNode).toMap() }
        map[MULTI_GOALS_DATA] = JsonUtils.MAPPER.valueToTree(mgArray)
        return map
    }

    private fun findMultiGoalsIndex(metrics: ArrayNode): Int {
        metrics.forEachIndexed { index, node ->
            if (node.isArray) {
                return index
            }
        }
        return -1
    }

    private fun transformTotals(totalsNode: JsonNode): JsonNode {
        val totals = mutableMapOf<String, JsonNode>()
        val mgArray = mutableListOf<Map<String, JsonNode>>()
        totalsNode.fields().forEach { e ->
            val key = e.key
            val value = e.value
            if (!value.isArray) {
                totals[key] = value
            } else {
                val multiGoalsHeader = key
                        .replace("\"", "")
                        .replace("[", "")
                        .replace("]", "")
                        .split(",")
                val metrics = value as ArrayNode
                metrics.mapTo(mgArray, { multiGoalsHeader.zip(it as ArrayNode).toMap() })
            }
        }
        totals[MULTI_GOALS_DATA] = JsonUtils.MAPPER.valueToTree(mgArray)
        return JsonUtils.MAPPER.valueToTree(totals)
    }
}
