package ru.yandex.direct.bannersystem.handle;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.databind.JavaType;
import org.apache.http.entity.ContentType;

import ru.yandex.direct.bannersystem.BsHostType;
import ru.yandex.direct.utils.JsonUtils;

/**
 * Ручка в БК, принимающая и отдающая результаты в JSON
 */
@ParametersAreNonnullByDefault
public class BsJsonHandleSpec<T, P> extends BsHandleSpec<T, P> {
    private final JavaType responseType;
    private final Class<P> responseClass;

    public BsJsonHandleSpec(String urlPath, BsHostType hostType, boolean hasPreProd) {
        this(urlPath, hostType, hasPreProd, null, null);
    }

    public BsJsonHandleSpec(String urlPath, BsHostType hostType, boolean hasPreProd, JavaType responseType) {
        this(urlPath, hostType, hasPreProd, responseType, null);
    }

    public BsJsonHandleSpec(String urlPath, BsHostType hostType, boolean hasPreProd, Class<P> responseClass) {
        this(urlPath, hostType, hasPreProd, null, responseClass);
    }

    private BsJsonHandleSpec(String urlPath, BsHostType hostType, boolean hasPreProd, @Nullable JavaType responseType,
                             @Nullable Class<P> responseClass) {
        super(urlPath, hostType, hasPreProd);

        this.responseClass = responseClass;
        this.responseType = responseType;
    }

    public <R> BsJsonHandleSpec<T, R> withResponseClass(JavaType responseType) {
        return new BsJsonHandleSpec<>(getUrlPath(), getHostType(), hasPreProd(), responseType);
    }

    @Override
    public String serializeRequestBody(Object bodyObj) {
        return JsonUtils.toJson(bodyObj);
    }

    @Override
    public P deserializeResponseBody(String responseBody) {
        if (responseType != null) {
            return JsonUtils.fromJson(responseBody, responseType);
        } else if (responseClass != null) {
            return JsonUtils.fromJson(responseBody, responseClass);
        } else {
            throw new IllegalStateException("type or class not provided");
        }
    }

    @Override
    public ContentType getRequestContentType() {
        return ContentType.APPLICATION_JSON;
    }
}
