package ru.yandex.direct.binlogclickhouse;

import java.time.Duration;
import java.util.Optional;

import ru.yandex.direct.binlog.reader.BinlogStateSet;
import ru.yandex.direct.binlog.reader.NamedBinlogState;
import ru.yandex.direct.utils.MonotonicClock;
import ru.yandex.direct.utils.MonotonicTime;
import ru.yandex.direct.utils.NanoTimeClock;

public class BinlogStateDelayedSaver implements BinlogStateSaver, AutoCloseable {
    private final MonotonicClock clock;
    private final BinlogStateSaver stateSaver;
    private final Duration stateSaveDelay;
    private MonotonicTime nextSaveTime;
    private BinlogStateSet stateSet;

    public BinlogStateDelayedSaver(BinlogStateSaver stateSaver, Duration stateSaveDelay) {
        this.clock = NanoTimeClock.CLOCK;
        this.stateSaver = stateSaver;
        this.stateSaveDelay = stateSaveDelay;
        this.nextSaveTime = clock.getTime();
        this.stateSet = new BinlogStateSet();
    }

    @Override
    public void saveStates(BinlogStateSet newStateSet) {
        for (NamedBinlogState newState : newStateSet) {
            // DIRECT-79441
            // Если синхронизируется кластер, в котором мало событий, возможна ситуация, что в одном шарде
            // за несколько минут были изменения, а в другом не было.
            if (!stateSet.getBySourceName(newState.getSourceName()).equals(Optional.of(newState.getState()))) {
                stateSet.add(newState);
            }
        }
        if (clock.getTime().isAfter(nextSaveTime)) {
            flush();
            nextSaveTime = clock.getTime().plus(stateSaveDelay);
        }
    }

    public void flush() {
        stateSaver.saveStates(stateSet);
        stateSet = new BinlogStateSet();
    }

    @Override
    public void close() {
        flush();
    }
}
