package ru.yandex.direct.binlogclickhouse;

import java.time.Duration;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZoneOffset;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

import com.github.shyiko.mysql.binlog.event.QueryEventData;
import com.github.shyiko.mysql.binlog.event.RowsQueryEventData;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.binlog.reader.MySQLSimpleRowIndexed;
import ru.yandex.direct.binlog.reader.MySQLSimpleRowsIndexed;
import ru.yandex.direct.binlogclickhouse.schema.DbChangeLogRecord;
import ru.yandex.direct.binlogclickhouse.schema.FieldValue;
import ru.yandex.direct.binlogclickhouse.schema.FieldValueList;
import ru.yandex.direct.binlogclickhouse.schema.Operation;
import ru.yandex.direct.mysql.MySQLBinlogConsumer;
import ru.yandex.direct.mysql.MySQLBinlogDataStreamer;
import ru.yandex.direct.mysql.MySQLSimpleData;
import ru.yandex.direct.mysql.MySQLUpdateData;
import ru.yandex.direct.mysql.MySQLUpdateRow;
import ru.yandex.direct.mysql.schema.KeySchema;
import ru.yandex.direct.mysql.schema.ServerSchema;
import ru.yandex.direct.mysql.schema.TableSchema;
import ru.yandex.direct.tracing.data.DirectTraceInfo;
import ru.yandex.direct.utils.CommonUtils;
import ru.yandex.direct.utils.MonotonicTime;
import ru.yandex.direct.utils.NanoTimeClock;

public class ClickHouseConsumer implements MySQLBinlogConsumer {
    private static final Logger logger = LoggerFactory.getLogger(ClickHouseConsumer.class);

    private final String source;
    private final Inserter inserter;

    private final Duration timeLimit;
    private LocalDateTime firstTransactionDateTime;
    private boolean streamerStopped = false;

    private MySQLBinlogDataStreamer streamer;
    private BinlogTransaction currentTransaction;
    private BinlogTransactionChangeAdder currentQuery;

    public ClickHouseConsumer(String source, Inserter inserter) {
        this(source, inserter, null);
    }

    public ClickHouseConsumer(String source, Inserter inserter, Duration timeLimit) {
        this.source = source;
        this.inserter = inserter;

        this.timeLimit = timeLimit;
        this.firstTransactionDateTime = null;
        this.streamerStopped = false;

        this.streamer = null;
        this.currentTransaction = null;
        this.currentQuery = null;
    }

    @Override
    public void onConnect(MySQLBinlogDataStreamer streamer) {
        if (this.streamer != null) {
            throw new IllegalStateException("Second connect when already connected");
        }
        this.streamer = streamer;
    }

    @Override
    public void onDisconnect(MySQLBinlogDataStreamer streamer) {
        /*
        disconnect может приходить еще до того как обработаны все записи
        из бинлога, так что затирать streamer нельзя.

        if (this.streamer != streamer) {
            throw new IllegalStateException("Disconnect from unexpected streamer");
        }
        this.streamer = null;
        */
    }

    @Override
    public void onDDL(String gtid, QueryEventData data, ServerSchema before, ServerSchema after) {
        // TODO akimov@: писать в кликхаус TRUNCATE и пр.
        //  (http://dev.mysql.com/doc/refman/5.7/en/sql-syntax-data-definition.html)
    }

    @Override
    public void onTransactionBegin(String gtid) {
        if (currentTransaction != null) {
            throw new IllegalStateException("Transaction begin (gtid=" + gtid +
                    "), while another transaction in progress (gtid=" + currentTransaction.getGtid() + ")"
            );
        }
        currentTransaction = new BinlogTransaction(gtid);
    }

    @Override
    public void onTransactionCommit(String gtid) {
        if (!gtid.equals(currentTransaction.getGtid())) {
            throw new IllegalStateException(
                    "Unexpected commit for gtid=" + gtid + ", expected gtid=" + currentTransaction.getGtid()
            );
        }
        if (withinAllowedTimeRange(currentTransaction)) {
            MonotonicTime start = NanoTimeClock.now();
            try {
                inserter.insert(new BinlogTransactionsBatch(source, streamer.getState(), currentTransaction));
            } finally {
                MonotonicTime stop = NanoTimeClock.now();
                if (stop.isAfter(start.plus(Duration.ofSeconds(10)))) {
                    logger.warn("Slow insertion ({} sec.) affects reading from {}",
                            stop.minus(start).toMillis() / 1000.0, streamer
                    );
                }
            }
        }
        currentTransaction = null;
        currentQuery = null;
    }

    private boolean withinAllowedTimeRange(BinlogTransaction transaction) {
        if (timeLimit == null) {
            return true;
        }

        if (transaction.getChanges().isEmpty()) {
            return true;
        }

        if (firstTransactionDateTime == null) {
            firstTransactionDateTime = transaction.getChanges().get(0).getDateTime();
            return true;
        }

        if (transaction.getChanges().get(0).getDateTime().isBefore(firstTransactionDateTime.plus(timeLimit))) {
            return true;
        }

        if (!streamerStopped) {
            if (logger.isInfoEnabled()) {
                logger.info("Successfully consumed {} of binlog, stopping streamer {}",
                        CommonUtils.formatApproxDuration(timeLimit), streamer);
            }
            streamer.stop();
            streamerStopped = true;
        }

        return false;
    }

    @Override
    public void onRowsQuery(RowsQueryEventData data, long timestamp) {
        if (currentTransaction == null) {
            throw new IllegalStateException("Rows query outside transaction");
        }
        currentQuery = currentTransaction.addQuery(
                DirectTraceInfo.extractIgnoringErrors(data.getQuery()),
                source,
                Instant.ofEpochMilli(timestamp).atZone(ZoneOffset.UTC).toLocalDateTime(),
                truncateQuery(data.getQuery(), 1 * 1024)
        );
    }

    @Override
    public void onUpdateRows(MySQLUpdateData data) {
        if (currentQuery == null) {
            if (currentTransaction == null) {
                throw new IllegalStateException("Rows query outside transaction");
            }
            currentQuery = currentTransaction.addQuery(
                    DirectTraceInfo.empty(),
                    source,
                    Instant.ofEpochMilli(data.getTimestamp()).atZone(ZoneOffset.UTC).toLocalDateTime(),
                    "-"
            );
        }
        if (!data.getTableMap().getTable().endsWith("_tmp")) {
            getPrimaryOrUniqueKey(data.getSchema()).ifPresent(key -> {
                Map<String, Integer> beforeNamesMap =
                        MySQLSimpleRowsIndexed.makeNameMap(data.getBeforeColumns().stream());
                for (MySQLUpdateRow row : data.getRows()) {
                    FieldValueList diffRow = FieldValueList.fromColumnDataListDiff(
                            row.getAfterUpdate(), row.getBeforeUpdate());
                    if (diffRow.size() != 0) {
                        currentQuery.addChange(
                                data.getTableMap().getDatabase(),
                                data.getTableMap().getTable(),
                                Operation.UPDATE,
                                DbChangeLogRecord.makePrimaryKey(
                                        key,
                                        new MySQLSimpleRowIndexed(beforeNamesMap, row.getBeforeUpdate())
                                ),
                                diffRow
                        );
                    }
                }
            });
        }
    }

    @Override
    public void onInsertRows(MySQLSimpleData data) {
        if (currentQuery == null) {
            if (currentTransaction == null) {
                throw new IllegalStateException("Rows query outside transaction");
            }
            currentQuery = currentTransaction.addQuery(
                    DirectTraceInfo.empty(),
                    source,
                    Instant.ofEpochMilli(data.getTimestamp()).atZone(ZoneOffset.UTC).toLocalDateTime(),
                    "-"
            );
        }
        if (!data.getTableMap().getTable().endsWith("_tmp")) {
            getPrimaryOrUniqueKey(data.getSchema()).ifPresent(key -> {
                for (MySQLSimpleRowIndexed row : MySQLSimpleRowsIndexed.fromSimpleRows(data.getRows())) {
                    currentQuery.addChange(
                            data.getTableMap().getDatabase(),
                            data.getTableMap().getTable(),
                            Operation.INSERT,
                            DbChangeLogRecord.makePrimaryKey(key, row),
                            new FieldValueList(row.stream()
                                    .filter(column -> key.getColumns().stream().noneMatch(
                                            keyColumn -> keyColumn.getName().equals(column.getSchema().getName())
                                    ))
                                    .map(column -> new FieldValue<>(
                                            column.getSchema().getName(),
                                            column.getValueAsString()
                                    ))
                                    .collect(Collectors.toList())
                            )
                    );
                }
            });
        }
    }

    @Override
    public void onDeleteRows(MySQLSimpleData data) {
        if (currentQuery == null) {
            if (currentTransaction == null) {
                throw new IllegalStateException("Rows query outside transaction");
            }
            currentQuery = currentTransaction.addQuery(
                    DirectTraceInfo.empty(),
                    source,
                    Instant.ofEpochMilli(data.getTimestamp()).atZone(ZoneOffset.UTC).toLocalDateTime(),
                    "-"
            );
        }
        if (!data.getTableMap().getTable().endsWith("_tmp")) {
            getPrimaryOrUniqueKey(data.getSchema()).ifPresent(key -> {
                for (MySQLSimpleRowIndexed row : MySQLSimpleRowsIndexed.fromSimpleRows(data.getRows())) {
                    currentQuery.addChange(
                            data.getTableMap().getDatabase(),
                            data.getTableMap().getTable(),
                            Operation.DELETE,
                            DbChangeLogRecord.makePrimaryKey(key, row),
                            FieldValueList.empty()
                    );
                }
            });
        }
    }

    private static Optional<KeySchema> getPrimaryOrUniqueKey(TableSchema tableSchema) {
        Optional<KeySchema> optionalKey = tableSchema.getPrimaryKey();

        if (!optionalKey.isPresent()) {
            optionalKey = tableSchema.getSoleUniqueKey();
        }

        return optionalKey;
    }

    private String truncateQuery(String query, int maxLength) {
        if (query.length() <= maxLength) {
            return query;
        } else {
            return "<truncated, actual_length=" + query.length() + "> " + query.substring(0, maxLength);
        }
    }
}
