package ru.yandex.direct.binlogclickhouse;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.binlogclickhouse.schema.DbChangeLog;
import ru.yandex.direct.binlogclickhouse.schema.DbChangeLogRecord;
import ru.yandex.direct.binlogclickhouse.schema.QueryLog;
import ru.yandex.direct.binlogclickhouse.schema.QueryLogRecord;
import ru.yandex.direct.utils.MonotonicTime;
import ru.yandex.direct.utils.NanoTimeClock;
import ru.yandex.direct.utils.Transient;
import ru.yandex.monlib.metrics.primitives.Counter;

import static ru.yandex.direct.solomon.SolomonUtils.SOLOMON_REGISTRY;

public class ClickHouseFlushInserter implements FlushableInserter {
    private static final Logger logger = LoggerFactory.getLogger(ClickHouseFlushInserter.class);

    private static final Counter SOLOMON_INSERTED_TRANSACTIONS = SOLOMON_REGISTRY.counter("inserted_transactions");
    private static final Counter SOLOMON_INSERTED_QUERIES = SOLOMON_REGISTRY.counter("inserted_queries");
    private static final Counter SOLOMON_INSERTED_ROWS = SOLOMON_REGISTRY.counter("inserted_rows");

    private DbChangeLog changeLog;
    private QueryLog queryLog;
    private BinlogStateSaver stateSaver;

    private DbChangeLog.Batch changeLogBatch;
    private QueryLog.Batch queryLogBatch;
    private BinlogTransactionsBatch lastStates;
    private long transactionsCount;


    public ClickHouseFlushInserter(DbChangeLog changeLog, QueryLog queryLog, BinlogStateSaver stateSaver) {
        this.changeLog = changeLog;
        this.queryLog = queryLog;
        this.stateSaver = stateSaver;
        reset();
    }

    @Override
    public void insert(BinlogTransactionsBatch transactions) {
        for (BinlogTransaction transaction : transactions.getTransactions()) {
            for (DbChangeLogRecord change : transaction.getChanges()) {
                changeLogBatch.add(change);
            }
            for (QueryLogRecord query : transaction.getQueries()) {
                queryLogBatch.add(query);
            }
        }
        if (!transactions.getStateSet().isEmpty()) {
            lastStates.add(new BinlogTransactionsBatch(transactions.getStateSet()));
        }
        transactionsCount += transactions.size();
    }

    @Override
    public void flush() {
        try {
            if (transactionsCount > 0) {
                MonotonicTime insertQueriesStart = NanoTimeClock.now();
                long queriesCount = queryLogBatch.execute();
                logger.info(
                        "inserted {} queries ({} transactions) in {} sec.",
                        queriesCount, transactionsCount,
                        NanoTimeClock.now().minus(insertQueriesStart).toMillis() / 1000.0
                );
                SOLOMON_INSERTED_QUERIES.add(queriesCount);

                MonotonicTime insertChangesStart = NanoTimeClock.now();
                long changesCount = changeLogBatch.execute();
                logger.info(
                        "inserted {} changes ({} transactions) in {} sec.",
                        changesCount, transactionsCount,
                        NanoTimeClock.now().minus(insertChangesStart).toMillis() / 1000.0
                );
                SOLOMON_INSERTED_ROWS.add(changesCount);

                SOLOMON_INSERTED_TRANSACTIONS.add(transactionsCount);
            }
        } finally {
            try {
                queryLogBatch.close();
            } finally {
                changeLogBatch.close();
            }
        }

        stateSaver.saveStates(lastStates.getStateSet());

        reset();
    }

    private void reset() {
        transactionsCount = 0;
        lastStates = new BinlogTransactionsBatch();
        try (Transient<DbChangeLog.Batch> holder = new Transient<>()) {
            holder.item = changeLogBatch = changeLog.createBatch();
            queryLogBatch = queryLog.createBatch();
            holder.success();
        }
    }
}
