package ru.yandex.direct.binlogclickhouse;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.binlogclickhouse.schema.DbChangeLog;
import ru.yandex.direct.binlogclickhouse.schema.DbChangeLogRecord;
import ru.yandex.direct.binlogclickhouse.schema.QueryLog;
import ru.yandex.direct.binlogclickhouse.schema.QueryLogRecord;
import ru.yandex.direct.utils.MonotonicTime;
import ru.yandex.direct.utils.NanoTimeClock;

public class ClickHouseInserter implements Inserter {
    private static final Logger logger = LoggerFactory.getLogger(ClickHouseInserter.class);

    private DbChangeLog changeLog;
    private QueryLog queryLog;
    private BinlogStateSaver stateSaver;

    public ClickHouseInserter(DbChangeLog changeLog, QueryLog queryLog, BinlogStateSaver stateSaver) {
        this.changeLog = changeLog;
        this.queryLog = queryLog;
        this.stateSaver = stateSaver;
    }

    @Override
    public void insert(BinlogTransactionsBatch transactions) {
        try (DbChangeLog.Batch changes = changeLog.createBatch()) {
            for (BinlogTransaction transaction : transactions.getTransactions()) {
                for (DbChangeLogRecord change : transaction.getChanges()) {
                    changes.add(change);
                }
            }
            MonotonicTime start = NanoTimeClock.now();
            long changesCount = changes.execute();
            logger.info("inserted {} changes ({} transactions) in {} sec.",
                    changesCount, transactions.size(), NanoTimeClock.now().minus(start).toMillis() / 1000.0
            );
        }

        try (QueryLog.Batch queries = queryLog.createBatch()) {
            for (BinlogTransaction transaction : transactions.getTransactions()) {
                for (QueryLogRecord query : transaction.getQueries()) {
                    queries.add(query);
                }
            }
            MonotonicTime start = NanoTimeClock.now();
            long queriesCount = queries.execute();
            logger.info("inserted {} queries ({} transactions) in {} sec.",
                    queriesCount, transactions.size(), NanoTimeClock.now().minus(start).toMillis() / 1000.0
            );
        }

        stateSaver.saveStates(transactions.getStateSet());
    }
}
