package ru.yandex.direct.binlogclickhouse;

import java.time.Duration;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.mysql.AsyncStreamer;
import ru.yandex.direct.mysql.MySQLBinlogDataStreamer;
import ru.yandex.direct.mysql.MySQLInstance;
import ru.yandex.direct.mysql.TransactionsCountWatcher;
import ru.yandex.direct.utils.AutoCloseableList;
import ru.yandex.direct.utils.Transient;

public class DbChangeLogWriter implements AutoCloseable {
    private static final Logger logger = LoggerFactory.getLogger(DbChangeLogWriter.class);

    private AsyncInserter asyncInserter;
    private AutoCloseableList<AsyncStreamer> streamers;
    private AutoCloseableList<BufferedInserter> streamerInserters;
    private AsyncStreamer.ExitMonitor exitMonitor;

    public DbChangeLogWriter(Inserter clickHouseInserter, AsyncStreamer.ExitMonitor exitMonitor) {
        this.exitMonitor = exitMonitor;
        streamers = new AutoCloseableList<>();
        streamerInserters = new AutoCloseableList<>();
        asyncInserter = new AsyncInserter(clickHouseInserter, 10);
    }

    public AsyncStreamer.ExitMonitor getExitMonitor() {
        return exitMonitor;
    }

    public long getTransactionsCount() {
        long count = 0;
        for (AsyncStreamer streamer : streamers) {
            count += ((TransactionsCountWatcher) streamer.getConsumer()).getTransactionsCount();
        }
        return count;
    }

    public void addAsyncStreamer(String source, MySQLBinlogDataStreamer streamer,
                                 MySQLInstance schemaReplicationMysql, Duration timeLimit) {
        try (
                Transient<BufferedInserter> streamerInserter = new Transient<>();
                Transient<AsyncStreamer> asyncStreamer = new Transient<>()
        ) {
            streamerInserter.item = new BufferedInserter(
                    asyncInserter, Duration.ofMinutes(1),
                    1000
            );
            asyncStreamer.item = new AsyncStreamer(
                    streamer,
                    new TransactionsCountWatcher(new ClickHouseConsumer(source, streamerInserter.item, timeLimit)),
                    schemaReplicationMysql,
                    exitMonitor
            );
            streamerInserters.add(streamerInserter);
            streamers.add(asyncStreamer);
        }
    }

    @Override
    public void close() {
        try (AsyncInserter ignored1 = asyncInserter) {
            logger.info("Closing after " + getTransactionsCount() + " transactions...");
            try (AutoCloseableList<BufferedInserter> ignored2 = streamerInserters) {
                try (AutoCloseableList<AsyncStreamer> ignored3 = streamers) {
                    logger.info("Stopping streamers...");
                    streamers.stream().forEach(streamer -> {
                        logger.info("Stopping streamer " + streamer + "...");
                        streamer.stop();
                    });
                    logger.info("Closing streamers...");
                }
                logger.info("Closing inserters...");
            }
        }
    }
}
