package ru.yandex.direct.binlogclickhouse;

import java.time.Duration;

import ru.yandex.direct.utils.MonotonicTime;
import ru.yandex.direct.utils.NanoTimeClock;

public class DelayedStateInserter implements Inserter, AutoCloseable {
    private final Inserter inserter;
    private final Duration stateSaveDelay;
    private BinlogTransactionsBatch lastUnsavedStates;
    private MonotonicTime nextSaveState;

    public DelayedStateInserter(Inserter inserter, Duration stateSaveDelay) {
        this.inserter = inserter;
        this.stateSaveDelay = stateSaveDelay;
        this.lastUnsavedStates = new BinlogTransactionsBatch();
        this.nextSaveState = null;
    }

    @Override
    public void insert(BinlogTransactionsBatch transactions) {
        if (nextSaveState == null) {
            nextSaveState = NanoTimeClock.now().plus(stateSaveDelay);
        }
        if (NanoTimeClock.now().isBefore(nextSaveState)) {
            lastUnsavedStates.add(new BinlogTransactionsBatch(transactions.popStateSet()));
            inserter.insert(transactions);
        } else {
            lastUnsavedStates.add(transactions);
            inserter.insert(lastUnsavedStates);
            lastUnsavedStates = new BinlogTransactionsBatch();
            nextSaveState = NanoTimeClock.now().plus(stateSaveDelay);
        }
    }

    @Override
    public void close() {
        if (lastUnsavedStates != null) {
            inserter.insert(lastUnsavedStates);
        }
    }
}
