package ru.yandex.direct.binlogclickhouse;

import java.time.Duration;
import java.util.Optional;
import java.util.concurrent.Semaphore;
import java.util.concurrent.TimeUnit;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.mysql.MySQLBinlogState;
import ru.yandex.direct.utils.Interrupts;
import ru.yandex.direct.utils.exception.RuntimeTimeoutException;

/**
 * На MDB (он же DBaaS, он же кликхаус-как-сервис) стоят низкие ограничения по памяти - не более 512 Мб от одного
 * клиента. Когда синхронизатор запускается и пытается одновременно взять стейт для всех шардов ppc, памяти не хватает и
 * падают все.
 */
@ParametersAreNonnullByDefault
public class LimitedBinlogStateGetter implements BinlogStateGetter {
    private final Semaphore semaphore;
    private final BinlogStateGetter forward;
    private final Duration lockWaitDuration;

    /**
     * @param forward          Запрашивалка стейта, в которую будут передаваться запросы.
     * @param semaphore        Ограничитель одновременно запрашиваемых стейтов.
     * @param lockWaitDuration Максимальное ожидание взятия семафора.
     */
    public LimitedBinlogStateGetter(BinlogStateGetter forward, Semaphore semaphore, Duration lockWaitDuration) {
        this.semaphore = semaphore;
        this.forward = forward;
        this.lockWaitDuration = lockWaitDuration;
    }

    @Override
    public Optional<MySQLBinlogState> getLastState(String source) {
        return Interrupts.failingGet(() -> {
            if (semaphore.tryAcquire(lockWaitDuration.toMillis(), TimeUnit.MILLISECONDS)) {
                try {
                    return forward.getLastState(source);
                } finally {
                    semaphore.release();
                }
            } else {
                throw new RuntimeTimeoutException(
                        "Failed to take binlog state getter semaphore for " + lockWaitDuration);
            }
        });
    }
}
