package ru.yandex.direct.binlogclickhouse;

import java.util.Collection;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;

import ru.yandex.direct.utils.AsyncConsumer;
import ru.yandex.direct.utils.BlockingChannel;
import ru.yandex.direct.utils.Checked;
import ru.yandex.direct.utils.Interrupts;
import ru.yandex.direct.utils.NamedThreadFactory;
import ru.yandex.direct.utils.Transient;

public class ParallelFlushInserter implements FlushableInserter, AutoCloseable {
    private BlockingChannel<FlushableInserter> insertersQueue;
    private AsyncConsumer<Future<BinlogTransactionsBatch>> flushManager;
    private ExecutorService executorService;

    private FlushableInserter currentInserter;
    private BinlogTransactionsBatch lastState;

    public ParallelFlushInserter(Collection<FlushableInserter> inserters) {
        try (
                Transient<BlockingChannel<FlushableInserter>> t1 = new Transient<>();
                Transient<AsyncConsumer<Future<BinlogTransactionsBatch>>> t2 = new Transient<>()
        ) {
            insertersQueue = t1.item = new BlockingChannel<>(inserters);
            flushManager = t2.item = new AsyncConsumer<>(
                    f -> asyncFlush(insertersQueue, f),
                    inserters.size() * 2,
                    "StateSaver"
            );
            currentInserter = insertersQueue.poll().orElseThrow(
                    () -> new IllegalStateException("Empty inserters collection: " + inserters)
            );
            lastState = new BinlogTransactionsBatch();
            executorService = Executors.newCachedThreadPool(new NamedThreadFactory("Inserter"));

            t2.success();
            t1.success();
        }
    }

    @Override
    public void insert(BinlogTransactionsBatch transactions) {
        // Состояние из пачки нужно удалять, чтообы из-за параллельного исполнения состояния не записывались в базу
        // раньше чем соответствующие данные
        lastState.add(new BinlogTransactionsBatch(transactions.popStateSet()));
        currentInserter.insert(transactions);
    }

    @Override
    public void flush() {
        FlushableInserter flushingInseter = currentInserter;
        BinlogTransactionsBatch flushingState = lastState;
        Interrupts.failingRun(() -> flushManager.accept(executorService.submit(() -> {
            try {
                flushingInseter.flush();
                // При исключении, flushingInseter возвращать в очередь не нужно, потому что он испорчен.
                insertersQueue.put(flushingInseter);
            } catch (Throwable exc) { // NOSONAR
                insertersQueue.close();
                throw exc;
            }
            return flushingState;
        })));
        try {
            currentInserter = insertersQueue.take();
        } catch (InterruptedException exc) {
            throw new IllegalStateException("Interrupted", exc);
        }
        lastState = new BinlogTransactionsBatch();
    }

    @Override
    public void close() {
        // insertersQueue используется в flushManager, поэтому insertersQueue закрываем последним
        try (BlockingChannel<FlushableInserter> ignored1 = insertersQueue) {
            try {
                flushManager.close();
            } finally {
                executorService.shutdownNow();
            }
        }
        // Недописанную пачку транзакций бросаем на пол, мало-ли что пошло не так.
    }

    private static void asyncFlush(BlockingChannel<FlushableInserter> insertersQueue,
                                   Future<BinlogTransactionsBatch> flushingFuture)
            throws InterruptedException {
        try {
            BinlogTransactionsBatch flushingState = flushingFuture.get();
            if (!flushingState.isEmpty()) {
                try {
                    FlushableInserter inserter = insertersQueue.take();
                    inserter.insert(flushingState);
                    inserter.flush();
                    // При исключении, flushingInseter возвращать в очередь не нужно, потому что он испорчен.
                    insertersQueue.put(inserter);
                } catch (Throwable exc) { // NOSONAR
                    insertersQueue.close();
                    throw exc;
                }
            }
        } catch (ExecutionException exc) {
            throw new Checked.CheckedException(exc);
        }
    }
}
