package ru.yandex.direct.binlogclickhouse;

import java.time.Duration;
import java.util.Collection;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.function.Consumer;

import ru.yandex.direct.binlog.reader.BinlogStateSet;
import ru.yandex.direct.binlog.reader.StateBound;
import ru.yandex.direct.utils.AsyncConsumer;
import ru.yandex.direct.utils.Checked;
import ru.yandex.direct.utils.Interrupts;
import ru.yandex.direct.utils.NamedThreadFactory;
import ru.yandex.direct.utils.Transient;

public class StateBoundParallelConsumer<T> implements Consumer<StateBound<T>>, AutoCloseable {
    private static final Duration CLOSE_TIMEOUT = Duration.ofMinutes(1);

    private final ArrayBlockingQueue<Consumer<T>> consumersQueue;
    private final AsyncConsumer<Future<BinlogStateSet>> futureStateSaver;
    private final ExecutorService executor;

    public StateBoundParallelConsumer(Collection<Consumer<T>> consumers, BinlogStateSaver stateSaver, int capacity) {
        this.consumersQueue = new ArrayBlockingQueue<>(consumers.size(), true, consumers);
        try (Transient<AsyncConsumer<Future<BinlogStateSet>>> holder = new Transient<>()) {
            this.futureStateSaver = holder.item = new AsyncConsumer<>(
                    future -> {
                        try {
                            stateSaver.saveStates(future.get());
                        } catch (ExecutionException exc) {
                            throw new Checked.CheckedException(exc);
                        }
                    },
                    capacity,
                    "StateSaver"
            );
            this.executor = Executors.newFixedThreadPool(consumers.size(), new NamedThreadFactory("UserActionWriter"));
            holder.success();
        }
    }

    @Override
    public void accept(StateBound<T> item) {
        Interrupts.failingRun(() -> futureStateSaver.accept(executor.submit(() -> {
            Consumer<T> consumer = consumersQueue.take();
            consumer.accept(item.getData());
            consumersQueue.put(consumer);
            return item.getStateSet();
        })));
    }

    @Override
    public void close() {
        try (AsyncConsumer<Future<BinlogStateSet>> ignored = futureStateSaver) {
            executor.shutdownNow();
            Interrupts.criticalTimeoutWait(CLOSE_TIMEOUT, executor::awaitTermination);
        }
    }
}
