package ru.yandex.direct.binlogclickhouse.schema;

import java.sql.Connection;
import java.sql.Date;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.time.LocalDateTime;
import java.util.Optional;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.binlog.reader.BinlogStateSet;
import ru.yandex.direct.binlogclickhouse.BinlogStateGetter;
import ru.yandex.direct.binlogclickhouse.BinlogStateSaver;
import ru.yandex.direct.clickhouse.ClickHouseTable;
import ru.yandex.direct.clickhouse.InsertStatement;
import ru.yandex.direct.mysql.MySQLBinlogState;
import ru.yandex.direct.utils.Checked;
import ru.yandex.direct.utils.JsonUtils;
import ru.yandex.direct.utils.MonotonicTime;
import ru.yandex.direct.utils.NanoTimeClock;

public class BinlogStateTable extends ClickHouseTable implements BinlogStateSaver, BinlogStateGetter {
    private static final Logger logger = LoggerFactory.getLogger(BinlogStateTable.class);

    public BinlogStateTable(Connection conn, String dbName, String tableName) {
        super(conn, dbName, tableName);
    }

    public void saveState(String source, MySQLBinlogState state) {
        saveState(source, state, LocalDateTime.now());
    }

    public void saveState(String source, MySQLBinlogState state, LocalDateTime dateTime) {
        saveState(source, state, dateTime, false);
    }

    public void resetState(String source) {
        saveState(source, null, LocalDateTime.now(), true);
    }

    public synchronized void saveState(String source, MySQLBinlogState state, LocalDateTime dateTime, boolean reset) {
        try (PreparedStatement statement = getConn().prepareStatement(
                "INSERT INTO " + getQuotedFullTableName() + " (source, date, datetime, state) VALUES (?, ?, ?, ?)"
        )) {
            statement.setString(1, source);
            statement.setDate(2, Date.valueOf(dateTime.toLocalDate()));
            statement.setTimestamp(3, Timestamp.valueOf(dateTime));
            if (reset) {
                if (state != null) {
                    throw new IllegalArgumentException("state must be null, when reset=true");
                }
                statement.setString(4, "");
            } else {
                statement.setString(4, JsonUtils.toJson(state));
            }
            statement.executeUpdate();
        } catch (SQLException exc) {
            throw new Checked.CheckedException(exc);
        }
    }

    @Override
    public void saveStates(BinlogStateSet stateSet) {
        saveStates(stateSet, LocalDateTime.now());
    }

    public void saveStates(BinlogStateSet stateSet, LocalDateTime dateTime) {
        if (!stateSet.isEmpty()) {
            MonotonicTime start = NanoTimeClock.now();
            try (InsertStatement insert = createInsertStatement(
                    BinlogStateSchema.SOURCE,
                    BinlogStateSchema.DATE,
                    BinlogStateSchema.DATETIME,
                    BinlogStateSchema.STATE
            )) {
                stateSet.forEach(stateRecord -> insert.newRow()
                        .setNext(BinlogStateSchema.SOURCE, stateRecord.getSourceName())
                        .setNext(BinlogStateSchema.DATE, dateTime.toLocalDate())
                        .setNext(BinlogStateSchema.DATETIME, dateTime)
                        .setNext(BinlogStateSchema.STATE, Optional.of(JsonUtils.toJson(stateRecord.getState())))
                );
                insert.execute();
            }
            MonotonicTime end = NanoTimeClock.now();
            logger.info("saved {} states in {} sec.", stateSet.size(), end.minus(start).toMillis() / 1000.0);
        }
    }

    @Override
    public synchronized Optional<MySQLBinlogState> getLastState(String source) {
        try (PreparedStatement statement = getConn().prepareStatement(
                "SELECT state" +
                        " FROM " + getQuotedFullTableName() +
                        " WHERE source=?" +
                        " ORDER BY date DESC, datetime DESC" +
                        " LIMIT 1"
        )) {
            statement.setString(1, source);
            ResultSet resultSet = statement.executeQuery();
            String savedState = null;

            if (resultSet.next()) {
                savedState = resultSet.getString(1);
                if (resultSet.next()) {
                    throw new IllegalStateException("Expected exactly one row, got more than one.");
                }
            }

            return savedState == null
                    ? Optional.empty()
                    : "".equals(savedState) // пустая строка означает, что состояние было сброшено (reset)
                    ? Optional.empty()
                    : Optional.of(JsonUtils.fromJson(savedState, MySQLBinlogState.class));
        } catch (SQLException exc) {
            throw new Checked.CheckedException(exc);
        }
    }
}
