package ru.yandex.direct.binlogclickhouse.schema;

import java.io.IOException;
import java.sql.Connection;
import java.sql.Date;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeFormatterBuilder;
import java.time.temporal.ChronoField;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Stream;

import com.fasterxml.jackson.core.type.TypeReference;

import ru.yandex.direct.clickhouse.ClickHouseTable;
import ru.yandex.direct.clickhouse.InsertStatement;
import ru.yandex.direct.clickhouse.ResponseFieldNameAcessor;
import ru.yandex.direct.tracing.data.DirectTraceInfo;
import ru.yandex.direct.utils.Checked;
import ru.yandex.direct.utils.JsonUtils;

// TODO akimov@: переименовать в RowsLog
public class DbChangeLog extends ClickHouseTable {

    public static final DateTimeFormatter DATE_TIME_FORMATTER =
            new DateTimeFormatterBuilder().append(DateTimeFormatter.ISO_LOCAL_DATE)
                    .appendLiteral(' ')
                    .appendPattern("HH:mm:ss")
                    .appendFraction(ChronoField.NANO_OF_SECOND, 0, 9, true)
                    .toFormatter();

    public static class Batch implements AutoCloseable {
        private InsertStatement insertStatement;

        public Batch(DbChangeLog changeLog) {
            insertStatement = changeLog.createInsertStatement(
                    DbChangeLogSchema.REQID,
                    DbChangeLogSchema.METHOD,
                    DbChangeLogSchema.SERVICE,
                    DbChangeLogSchema.SOURCE,
                    DbChangeLogSchema.DB,
                    DbChangeLogSchema.TABLE,
                    DbChangeLogSchema.OPERATION,
                    DbChangeLogSchema.GTID,
                    DbChangeLogSchema.GTID_SRC,
                    DbChangeLogSchema.GTID_SCN,
                    DbChangeLogSchema.QUERY_SEQ_NUM,
                    DbChangeLogSchema.CHANGE_SEQ_NUM,
                    DbChangeLogSchema.PRIMARY_KEY,
                    DbChangeLogSchema.PRIMARY_KEY_SCHEMA,
                    DbChangeLogSchema.DATE,
                    DbChangeLogSchema.DATETIME,
                    DbChangeLogSchema.ROW_NAMES,
                    DbChangeLogSchema.ROW_VALUES,
                    DbChangeLogSchema.ROW_NULLITIES
            );
        }

        public void add(DbChangeLogRecord record) {
            insertStatement.newRow()
                    .setNext(DbChangeLogSchema.REQID, record.getDirectTraceInfo().getReqId())
                    .setNext(DbChangeLogSchema.METHOD, record.getDirectTraceInfo().getMethod())
                    .setNext(DbChangeLogSchema.SERVICE, record.getDirectTraceInfo().getService())
                    .setNext(DbChangeLogSchema.SOURCE, record.getSource())
                    .setNext(DbChangeLogSchema.DB, record.getDb())
                    .setNext(DbChangeLogSchema.TABLE, record.getTable())
                    .setNext(DbChangeLogSchema.OPERATION, record.getOperation())
                    .setNext(DbChangeLogSchema.GTID, record.getGtid())
                    .setNext(DbChangeLogSchema.GTID_SRC, record.getGtidSrc())
                    .setNext(DbChangeLogSchema.GTID_SCN, record.getGtidScn())
                    .setNext(DbChangeLogSchema.QUERY_SEQ_NUM, record.getQuerySeqNum())
                    .setNext(DbChangeLogSchema.CHANGE_SEQ_NUM, record.getChangeSeqNum())
                    .setNext(DbChangeLogSchema.PRIMARY_KEY, String.join(FIELD_SEPARATOR,
                            record.getPrimaryKey().getValues()))
                    .setNext(DbChangeLogSchema.PRIMARY_KEY_SCHEMA, String.join(FIELD_SEPARATOR,
                            record.getPrimaryKey().getNames()))
                    .setNext(DbChangeLogSchema.DATE, record.getDateTime().toLocalDate())
                    .setNext(DbChangeLogSchema.DATETIME, record.getDateTime())
                    .setNext(DbChangeLogSchema.ROW_NAMES, record.getRow().getNames())
                    .setNext(DbChangeLogSchema.ROW_VALUES, record.getRow().getValuesAsNonNullStrings())
                    .setNext(DbChangeLogSchema.ROW_NULLITIES, record.getRow().getNullities());
        }

        public long execute() {
            insertStatement.execute();
            return insertStatement.getRowsCount();
        }

        @Override
        public void close() {
            insertStatement.close();
        }
    }

    private static final String FIELD_SEPARATOR = ",";
    private static final String FIELD_SEPARATOR_REGEX = ",";

    public DbChangeLog(Connection conn, String dbName, String tableName) {
        super(conn, dbName, tableName);
    }

    public Stream<DbChangeLogRecord> select(String query) throws SQLException, IOException {
        /*
        Используется createClickHouseStatement вместо prepareStatement, потому что последний не
        умеет десериализовывать массивы, а они нужны чтобы читать Nested-поля.
         */
        try (Statement statement = getConn().createStatement()) {
            return DbChangeLog.fromResultSet(statement.executeQuery(query));
        }
    }

    public static Stream<DbChangeLogRecord> fromResultSet(ResultSet resultSet) {
        return new ResponseFieldNameAcessor(resultSet).getData()
                .map(rawRecord -> {
                    // @zhur настоял на том, чтобы строки писались как есть, без экранирования и квотирования,
                    // для простоты поиска из командной строчки.
                    // Поэтому запятые в именах или значениях полей испортят десериализацию.
                    String[] primaryKeySchema = rawRecord.get("primary_key_schema").split(FIELD_SEPARATOR_REGEX);
                    // Нужно, чтобы количество значений совпадало с количеством имен полей, поэтому разбиваем строку
                    // так, чтобы кусочков получилось не больше чем в ключе.
                    String[] primaryKey = rawRecord.get("primary_key").split(FIELD_SEPARATOR_REGEX,
                            primaryKeySchema.length);
                    if (primaryKeySchema.length != primaryKey.length) {
                        // Упс, встретилась запятая в имени поля, переразбиваем так, чтобы количество элементов совпало.
                        primaryKeySchema = rawRecord.get("primary_key_schema")
                                .split(FIELD_SEPARATOR_REGEX, primaryKey.length);
                    }

                    return new DbChangeLogRecord(
                            new DirectTraceInfo(
                                    DbChangeLogSchema.REQID.getType().fromClickHouseString(rawRecord.get("reqid")),
                                    DbChangeLogSchema.SERVICE.getType().fromClickHouseString(rawRecord.get("service")),
                                    DbChangeLogSchema.METHOD.getType().fromClickHouseString(rawRecord.get("method"))
                            ),
                            rawRecord.get("source"),
                            rawRecord.get("db"),
                            rawRecord.get("table"),
                            Operation.valueOf(rawRecord.get("operation")),
                            rawRecord.get("gtid"),
                            rawRecord.get("gtid_src"),
                            Long.parseLong(rawRecord.get("gtid_scn")),
                            Integer.parseInt(rawRecord.get("query_seq_num")),
                            Integer.parseInt(rawRecord.get("change_seq_num")),
                            FieldValueList.zip(Arrays.asList(primaryKeySchema), Arrays.asList(primaryKey)),
                            LocalDateTime.parse(rawRecord.get("datetime"), DATE_TIME_FORMATTER),
                            FieldValueList.zip(
                                    JsonUtils.fromJson(rawRecord.get("row.name"), new TypeReference<List<String>>() {
                                    }),
                                    JsonUtils.fromJson(rawRecord.get("row.value"), new TypeReference<List<String>>() {
                                    }),
                                    JsonUtils.fromJson(rawRecord.get("row.is_null"),
                                            new TypeReference<List<Boolean>>() {
                                            })
                            )
                    );
                });

    }

    public Batch createBatch() {
        return new Batch(this);
    }

    public Map<String, LocalDateTime> getSourceRecencySince(LocalDate date) {
        Map<String, LocalDateTime> recency = new HashMap<>();

        try (PreparedStatement statement = getConn().prepareStatement(
                "SELECT source, max(datetime)" +
                        " FROM " + getQuotedFullTableName() +
                        " WHERE date >= ?" +
                        " GROUP BY source"
        )) {
            statement.setDate(1, Date.valueOf(date));
            try (ResultSet resultSet = statement.executeQuery()) {
                while (resultSet.next()) {
                    recency.put(
                            resultSet.getString(1),
                            resultSet.getTimestamp(2).toLocalDateTime()
                    );
                }
            }
        } catch (SQLException exc) {
            throw new Checked.CheckedException(exc);
        }

        return recency;
    }
}
