package ru.yandex.direct.binlogclickhouse.schema;

import java.sql.Connection;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.util.Arrays;
import java.util.List;
import java.util.Optional;
import java.util.OptionalLong;

import com.fasterxml.jackson.core.type.TypeReference;

import ru.yandex.direct.clickhouse.NestedField;
import ru.yandex.direct.clickhouse.SimpleField;
import ru.yandex.direct.clickhouse.TableSchema;
import ru.yandex.direct.clickhouse.types.BooleanClickHouseType;
import ru.yandex.direct.clickhouse.types.DateClickHouseType;
import ru.yandex.direct.clickhouse.types.DateTimeClickHouseType;
import ru.yandex.direct.clickhouse.types.EnumClickHouseType;
import ru.yandex.direct.clickhouse.types.IntegerClickHouseType;
import ru.yandex.direct.clickhouse.types.LongClickHouseType;
import ru.yandex.direct.clickhouse.types.OptionalLongClickHouseType;
import ru.yandex.direct.clickhouse.types.OptionalStringClickHouseType;
import ru.yandex.direct.clickhouse.types.StringClickHouseType;

public class DbChangeLogSchema extends TableSchema {
    public static final SimpleField<OptionalLong> REQID =
            new SimpleField<>("reqid", new OptionalLongClickHouseType("Int64", 0, true));
    public static final SimpleField<Optional<String>> METHOD = new SimpleField<>("method",
            new OptionalStringClickHouseType(""));
    public static final SimpleField<Optional<String>> SERVICE = new SimpleField<>("service",
            new OptionalStringClickHouseType(""));
    public static final SimpleField<String> SOURCE = new SimpleField<>("source", new StringClickHouseType());
    public static final SimpleField<String> DB = new SimpleField<>("db", new StringClickHouseType());
    public static final SimpleField<String> TABLE = new SimpleField<>("table", new StringClickHouseType());
    public static final SimpleField<Operation> OPERATION = new SimpleField<>("operation", new EnumClickHouseType<>(
            "Enum8", Operation.class));
    public static final SimpleField<String> GTID = new SimpleField<>("gtid", new StringClickHouseType());
    public static final SimpleField<String> GTID_SRC = new SimpleField<>("gtid_src", new StringClickHouseType());
    public static final SimpleField<Long> GTID_SCN = new SimpleField<>("gtid_scn", new LongClickHouseType("Int64"));
    public static final SimpleField<Integer> QUERY_SEQ_NUM = new SimpleField<>("query_seq_num",
            new IntegerClickHouseType("UInt32"));
    public static final SimpleField<Integer> CHANGE_SEQ_NUM = new SimpleField<>("change_seq_num",
            new IntegerClickHouseType("UInt32"));
    public static final SimpleField<String> PRIMARY_KEY = new SimpleField<>("primary_key", new StringClickHouseType());
    public static final SimpleField<String> PRIMARY_KEY_SCHEMA = new SimpleField<>("primary_key_schema",
            new StringClickHouseType());
    public static final SimpleField<LocalDateTime> DATETIME = new SimpleField<>("datetime",
            new DateTimeClickHouseType());
    public static final SimpleField<LocalDate> DATE = new SimpleField<>("date", new DateClickHouseType());

    public static final NestedField ROW = new NestedField("row");
    public static final NestedField.SubFieldArray<String> ROW_NAMES = ROW.add("name", new StringClickHouseType(),
            new TypeReference<List<String>>() {
    });
    public static final NestedField.SubFieldArray<String> ROW_VALUES = ROW.add("value", new StringClickHouseType(),
            new TypeReference<List<String>>() {
    });
    public static final NestedField.SubFieldArray<Boolean> ROW_NULLITIES = ROW.add("is_null",
            new BooleanClickHouseType(), new TypeReference<List<Boolean>>() {
    });

    public static final SimpleField<Integer> COLLAPSING_SIGN = new SimpleField<>("collapsing_sign",
            new IntegerClickHouseType("Int8 DEFAULT 1"));

    public DbChangeLogSchema(String dbName, String tableName) {
        /*
        При падениях binlog-clickhouse в кликхаусе могут появляться дубликаты операций.
        С ними можно бороться несколькими способами, используя подвиды MergeTree:

        - CollapsingMergeTree - добавляем в таблички поле sign и мониторим повторные вставки грепом
          в логах кликхауса (<Warning> CollapsingSortedBlockInputStream: Incorrect data: number of rows
          with sign = 1 (2) differs with number of rows with sign = -1 (0) by more than one (for key:
          '8a1b066a-9550-11e6-9135-4851b74386c7:6', 0).);

        - SummingMergeTree - добавляем в таблички суммируемое поле, инкрементируем его при каждой повторной
          вставке. Мониторить повторные вставки - понятно как, select-ами. Но есть опасность, clickhouse может
          выбрасывать строки, если суммируемое поле стало равно нулю, а это может случиться при переполнении;

        - AggregatingMergeTree - не добавляем ни одного аггрегирующего поля, тогда в табличку можно инсертить
          напрямую. Просто и изящно, но вообще нет возможности мониторить повторные вставки;

        - Есть недокументированное ReplacingMergeTree, по поведению эквивалентно варианту с AggregatingMergeTree.

        Используется первый вариант.
         */
        super(
                dbName,
                tableName,
                Arrays.asList(
                        REQID,
                        METHOD,
                        SERVICE,
                        SOURCE,
                        DB,
                        TABLE,
                        OPERATION,
                        GTID,
                        GTID_SRC,
                        GTID_SCN,
                        QUERY_SEQ_NUM,
                        CHANGE_SEQ_NUM,
                        PRIMARY_KEY,
                        PRIMARY_KEY_SCHEMA,
                        DATETIME,
                        DATE,
                        ROW,
                        COLLAPSING_SIGN
                ),
                "CollapsingMergeTree",
                Arrays.asList(
                        DATE.getName(),
                        "cityHash64(" + PRIMARY_KEY.getName() + ")",
                        // Без date в ключе гранулярность поиска по дате была бы слишком большая,
                        // потому что clickhouse постепенно аггрегирует данные в куски размером до одного месяца.
                        "(" + String.join(", ",
                                DATE.getName(),
                                DB.getName(),
                                TABLE.getName(),
                                PRIMARY_KEY.getName(),
                                GTID.getName(),
                                CHANGE_SEQ_NUM.getName(),
                                "cityHash64(" + PRIMARY_KEY.getName() + ")"
                        ) + ")",
                        "8192",
                        COLLAPSING_SIGN.getName()
                )
        );
    }

    public DbChangeLog connect(Connection conn) {
        return new DbChangeLog(conn, getDbName(), getTableName());
    }
}
