package ru.yandex.direct.binlogclickhouse.schema;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

public class FieldValue<T> {
    private final String name;
    private final T value;

    public FieldValue(String name, T value) {
        this.name = name;
        this.value = value;
    }

    public FieldValue(Map.Entry<String, T> entry) {
        this(entry.getKey(), entry.getValue());
    }

    public static <T> List<FieldValue<T>> zip(List<String> names, List<T> values) {
        if (names.size() != values.size()) {
            throw new IllegalArgumentException(
                    "Names and values length mismatch (" + names.size() + "!=" + values.size() + ")"
            );
        }
        List<FieldValue<T>> fields = new ArrayList<>();
        for (int i = 0; i < names.size(); i++) {
            fields.add(new FieldValue<>(names.get(i), values.get(i)));
        }
        return fields;
    }

    public static <T> List<FieldValue<T>> zip(List<String> names, List<T> values, List<Boolean> nullities) {
        if (names.size() != values.size()) {
            throw new IllegalArgumentException(
                    "Names and values length mismatch (" + names.size() + "!=" + values.size() + ")"
            );
        }
        if (names.size() != nullities.size()) {
            throw new IllegalArgumentException(
                    "Names and nullities length mismatch (" + names.size() + "!=" + nullities.size() + ")"
            );
        }
        List<FieldValue<T>> fields = new ArrayList<>();
        for (int i = 0; i < names.size(); i++) {
            fields.add(new FieldValue<>(names.get(i), nullities.get(i) ? null : values.get(i)));
        }
        return fields;
    }

    public String getName() {
        return name;
    }

    public T getValue() {
        return value;
    }

    public String getValueAsNonNullString() {
        if (value == null) {
            return "";
        } else {
            return value.toString();
        }
    }

    public boolean valueIsNull() {
        return value == null;
    }

    @Override
    public String toString() {
        return "FieldValue{" +
                "name='" + name + '\'' +
                ", value=" + (value instanceof String ? "\'" + value + '\'' : value) +
                '}';
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }

        FieldValue<?> that = (FieldValue<?>) o;

        if (!name.equals(that.name)) {
            return false;
        }
        return value != null ? value.equals(that.value) : that.value == null;

    }

    @Override
    public int hashCode() {
        int result = name.hashCode();
        result = 31 * result + (value != null ? value.hashCode() : 0);
        return result;
    }
}
