package ru.yandex.direct.binlogclickhouse.schema;

import java.util.Arrays;
import java.util.Collections;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import ru.yandex.direct.mysql.MySQLColumnData;

public class FieldValueList {
    private final List<FieldValue<String>> fieldsValues;

    public FieldValueList(List<FieldValue<String>> fieldsValues) {
        this.fieldsValues = fieldsValues;
    }

    @SafeVarargs
    public FieldValueList(FieldValue<String>... fieldsValues) {
        this(Arrays.asList(fieldsValues));
    }

    public List<FieldValue<String>> getFieldsValues() {
        return fieldsValues;
    }

    public static FieldValueList empty() {
        return new FieldValueList(Collections.emptyList());
    }

    public static FieldValueList zip(List<String> names, List<String> values) {
        return new FieldValueList(FieldValue.zip(names, values));
    }

    public static FieldValueList zip(List<String> names, List<String> values, List<Boolean> nullities) {
        return new FieldValueList(FieldValue.zip(names, values, nullities));
    }

    public static FieldValueList fromColumnDataList(List<MySQLColumnData> fields) {
        return new FieldValueList(
                fields.stream()
                        .map(f -> new FieldValue<>(f.getSchema().getName(), f.getValueAsString()))
                        .collect(Collectors.toList())
        );
    }

    private static FieldValueList fromStream(Stream<MySQLColumnData> source) {
        return new FieldValueList(source
                .map(f -> new FieldValue<>(f.getSchema().getName(), f.getValueAsString()))
                .collect(Collectors.toList()));
    }

    /**
     * Список ячеек первого кортежа, в котором отсутствуют ячейки, совпадающие с ячейками из второго кортежа.
     *
     * @param source  Исходный кортеж.
     * @param exclude Кортеж, значения которого следует исключить из исходного кортежа.
     * @return Список полей
     */
    public static FieldValueList fromColumnDataListDiff(List<MySQLColumnData> source, List<MySQLColumnData> exclude) {
        Set<MySQLColumnData> excludeSet = new HashSet<>(exclude);
        return fromStream(source.stream()
                .filter(field -> !excludeSet.contains(field)));
    }

    /**
     * Список ячеек первого кортежа, в котором отсутствуют ячейки, совпадающие с ячейками из второго кортежа.
     * Также будут исключены все поля, названия колонок которых совпадают с указанными в отдельном множестве.
     *
     * @param source            Исходный кортеж.
     * @param exclude           Кортеж, значения которого следует исключить из исходного кортежа.
     * @param ignoredFieldNames Множество названий колонок, которые будут отсутствовать в результате независимо
     *                          от того, отличаются они в source и exclude или нет.
     * @return Список полей
     */
    public static FieldValueList fromColumnDataListDiffFilter(
            List<MySQLColumnData> source, List<MySQLColumnData> exclude,
            Set<String> ignoredFieldNames
    ) {
        Set<MySQLColumnData> excludeSet = new HashSet<>(exclude);
        return fromStream(source.stream()
                .filter(field -> !excludeSet.contains(field)
                        && !ignoredFieldNames.contains(field.getSchema().getName())));
    }

    /**
     * Список ячеек кортежа, в котором будут исключены все поля, названия которых совпадают с указанными в отдельном
     * множестве.
     *
     * @param source            Исходный кортеж.
     * @param ignoredFieldNames Множество названий колонок, которые будут отсутствовать в результате независимо
     *                          от того, отличаются они в source и exclude или нет.
     * @return Список полей
     */
    public static FieldValueList fromColumnDataListFilter(List<MySQLColumnData> source, Set<String> ignoredFieldNames) {
        return fromStream(source.stream()
                .filter(field -> !ignoredFieldNames.contains(field.getSchema().getName())));
    }

    public List<String> getNames() {
        return fieldsValues.stream().map(FieldValue::getName).collect(Collectors.toList());
    }

    public List<String> getValues() {
        return fieldsValues.stream().map(FieldValue::getValue).collect(Collectors.toList());
    }

    public List<String> getValuesAsNonNullStrings() {
        return fieldsValues.stream().map(FieldValue::getValueAsNonNullString).collect(Collectors.toList());
    }

    public List<Boolean> getNullities() {
        return fieldsValues.stream().map(FieldValue::valueIsNull).collect(Collectors.toList());
    }

    public int size() {
        return fieldsValues.size();
    }

    @Override
    public String toString() {
        return "FieldValueList{" + fieldsValues + '}';
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        FieldValueList that = (FieldValueList) o;
        return Objects.equals(fieldsValues, that.fieldsValues);
    }

    @Override
    public int hashCode() {
        return Objects.hash(fieldsValues);
    }

    /**
     * Создаёт ассоциативный массив из названий полей в сериализованные значения полей.
     * <p>
     * При итерации ключей и значений они будут идти в том же порядке, в котором шли в оригинальном списке. Поэтому
     * возвращается {@literal LinkedHashMap} вместо обычного {@literal Map}.
     */
    @SuppressWarnings("squid:S1319")
    public LinkedHashMap<String, String> toMap() {
        // load factor по умолчанию - 0.75; 1 / 0.75 == 4 / 3
        LinkedHashMap<String, String> result = new LinkedHashMap<>(fieldsValues.size() * 4 / 3 + 1);
        for (FieldValue fieldValue : fieldsValues) {
            result.put(fieldValue.getName(), (String) fieldValue.getValue());
        }
        return result;
    }
}
