package ru.yandex.direct.binlogclickhouse.schema;

import java.io.IOException;
import java.sql.Connection;
import java.sql.Date;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeFormatterBuilder;
import java.time.temporal.ChronoField;
import java.util.HashMap;
import java.util.Map;
import java.util.stream.Stream;

import ru.yandex.direct.clickhouse.ClickHouseTable;
import ru.yandex.direct.clickhouse.InsertStatement;
import ru.yandex.direct.clickhouse.ResponseFieldNameAcessor;
import ru.yandex.direct.tracing.data.DirectTraceInfo;
import ru.yandex.direct.utils.Checked;

public class QueryLog extends ClickHouseTable {
    public static final DateTimeFormatter DATE_TIME_FORMATTER =
            new DateTimeFormatterBuilder().append(DateTimeFormatter.ISO_LOCAL_DATE)
                    .appendLiteral(' ')
                    .appendPattern("HH:mm:ss")
                    .appendFraction(ChronoField.NANO_OF_SECOND, 0, 9, true)
                    .toFormatter();

    public static class Batch implements AutoCloseable {
        private final InsertStatement insertStatement;

        public Batch(QueryLog changeLog) {
            insertStatement = changeLog.createInsertStatement(
                    QueryLogSchema.REQID,
                    QueryLogSchema.METHOD,
                    QueryLogSchema.SERVICE,
                    QueryLogSchema.SOURCE,
                    QueryLogSchema.GTID,
                    QueryLogSchema.GTID_SRC,
                    QueryLogSchema.GTID_SCN,
                    QueryLogSchema.QUERY_SEQ_NUM,
                    QueryLogSchema.DATE,
                    QueryLogSchema.DATETIME,
                    QueryLogSchema.QUERY
            );
        }

        public void add(QueryLogRecord record) {
            insertStatement.newRow()
                    .setNext(QueryLogSchema.REQID, record.getDirectTraceInfo().getReqId())
                    .setNext(QueryLogSchema.METHOD, record.getDirectTraceInfo().getMethod())
                    .setNext(QueryLogSchema.SERVICE, record.getDirectTraceInfo().getService())
                    .setNext(QueryLogSchema.SOURCE, record.getSource())
                    .setNext(QueryLogSchema.GTID, record.getGtid())
                    .setNext(QueryLogSchema.GTID_SRC, record.getGtidSrc())
                    .setNext(QueryLogSchema.GTID_SCN, record.getGtidScn())
                    .setNext(QueryLogSchema.QUERY_SEQ_NUM, record.getQuerySeqNum())
                    .setNext(QueryLogSchema.DATE, record.getDateTime().toLocalDate())
                    .setNext(QueryLogSchema.DATETIME, record.getDateTime())
                    .setNext(QueryLogSchema.QUERY, record.getQuery());
        }

        public long execute() {
            insertStatement.execute();
            return insertStatement.getRowsCount();
        }

        @Override
        public void close() {
            insertStatement.close();
        }
    }

    public QueryLog(Connection conn, String dbName, String tableName) {
        super(conn, dbName, tableName);
    }

    public Stream<QueryLogRecord> select(String query) throws SQLException, IOException {
        /*
        Используется createClickHouseStatement вместо prepareStatement, потому что последний не
        умеет десериализовывать массивы, а они нужны чтобы читать Nested-поля.
         */
        try (Statement statement = getConn().createStatement()) {
            return QueryLog.fromResultSet(statement.executeQuery(query));
        }
    }

    public static Stream<QueryLogRecord> fromResultSet(ResultSet response) throws IOException {
        return new ResponseFieldNameAcessor(response).getData()
                .map(rawRecord ->
                        new QueryLogRecord(
                                new DirectTraceInfo(
                                        QueryLogSchema.REQID.getType().fromClickHouseString(rawRecord.get("reqid")),
                                        QueryLogSchema.SERVICE.getType().fromClickHouseString(rawRecord.get("service")),
                                        QueryLogSchema.METHOD.getType().fromClickHouseString(rawRecord.get("method"))
                                ),
                                rawRecord.get("source"),
                                rawRecord.get("gtid"),
                                rawRecord.get("gtid_src"),
                                Long.parseLong(rawRecord.get("gtid_scn")),
                                Integer.parseInt(rawRecord.get("query_seq_num")),
                                LocalDateTime.parse(
                                        rawRecord.get("datetime"),
                                        DATE_TIME_FORMATTER
                                ),
                                rawRecord.get("query")
                        )
                );
    }

    public Batch createBatch() {
        return new Batch(this);
    }

    public Map<String, LocalDateTime> getSourceRecencySince(LocalDate date) {
        Map<String, LocalDateTime> recency = new HashMap<>();

        try (PreparedStatement statement = getConn().prepareStatement(
                "SELECT source, max(datetime)" +
                        " FROM " + getQuotedFullTableName() +
                        " WHERE date >= ?" +
                        " GROUP BY source"
        )) {
            statement.setDate(1, Date.valueOf(date));
            try (ResultSet resultSet = statement.executeQuery()) {
                while (resultSet.next()) {
                    recency.put(
                            resultSet.getString(1),
                            resultSet.getTimestamp(2).toLocalDateTime()
                    );
                }
            }
        } catch (SQLException exc) {
            throw new Checked.CheckedException(exc);
        }

        return recency;
    }
}
