package ru.yandex.direct.binlogclickhouse.schema;

import java.sql.Connection;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.util.Arrays;
import java.util.Optional;
import java.util.OptionalLong;

import ru.yandex.direct.clickhouse.SimpleField;
import ru.yandex.direct.clickhouse.TableSchema;
import ru.yandex.direct.clickhouse.types.DateClickHouseType;
import ru.yandex.direct.clickhouse.types.DateTimeClickHouseType;
import ru.yandex.direct.clickhouse.types.IntegerClickHouseType;
import ru.yandex.direct.clickhouse.types.LongClickHouseType;
import ru.yandex.direct.clickhouse.types.OptionalLongClickHouseType;
import ru.yandex.direct.clickhouse.types.OptionalStringClickHouseType;
import ru.yandex.direct.clickhouse.types.StringClickHouseType;

public class QueryLogSchema extends TableSchema {
    public static final SimpleField<OptionalLong> REQID =
            new SimpleField<>("reqid", new OptionalLongClickHouseType("Int64", 0, true));
    public static final SimpleField<Optional<String>> METHOD =
            new SimpleField<>("method", new OptionalStringClickHouseType(""));
    public static final SimpleField<Optional<String>> SERVICE =
            new SimpleField<>("service", new OptionalStringClickHouseType(""));
    public static final SimpleField<String> SOURCE = new SimpleField<>("source", new StringClickHouseType());
    public static final SimpleField<String> GTID = new SimpleField<>("gtid", new StringClickHouseType());
    public static final SimpleField<String> GTID_SRC = new SimpleField<>("gtid_src", new StringClickHouseType());
    public static final SimpleField<Long> GTID_SCN =
            new SimpleField<>("gtid_scn", new LongClickHouseType("Int64"));
    public static final SimpleField<Integer> QUERY_SEQ_NUM =
            new SimpleField<>("query_seq_num", new IntegerClickHouseType("UInt32"));
    public static final SimpleField<LocalDate> DATE = new SimpleField<>("date", new DateClickHouseType());
    public static final SimpleField<LocalDateTime> DATETIME =
            new SimpleField<>("datetime", new DateTimeClickHouseType());
    public static final SimpleField<String> QUERY = new SimpleField<>("query", new StringClickHouseType());
    public static final SimpleField<Integer> COLLAPSING_SIGN =
            new SimpleField<>("collapsing_sign", new IntegerClickHouseType("Int8 DEFAULT 1"));

    public QueryLogSchema(String dbName, String tableName) {
        /*
        CollapsingMergeTree используется для удаления дубликатов.
        См. подробности в DbChangeLog.createTable().
         */
        super(
                dbName,
                tableName,
                Arrays.asList(
                        REQID,
                        METHOD,
                        SERVICE,
                        SOURCE,
                        GTID,
                        GTID_SRC,
                        GTID_SCN,
                        QUERY_SEQ_NUM,
                        DATE,
                        DATETIME,
                        QUERY,
                        COLLAPSING_SIGN
                ),
                "CollapsingMergeTree",
                Arrays.asList(
                        DATE.getName(),
                        // Без date в ключе гранулярность поиска по дате была бы слишком большая,
                        // clickhouse постепенно аггрегирует данные до месяца.
                        "(" + DATE.getName() + ", " + GTID.getName() + ", " + QUERY_SEQ_NUM.getName() + ")",
                        "8192",
                        COLLAPSING_SIGN.getName()
                )
        );
    }

    public QueryLog connect(Connection conn) {
        return new QueryLog(conn, getDbName(), getTableName());
    }
}
