package ru.yandex.direct.binlog.model;

import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.ZoneOffset;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.function.Consumer;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.annotation.JsonGetter;
import com.fasterxml.jackson.annotation.JsonIgnore;
import com.google.common.base.Preconditions;

import static org.apache.commons.lang3.StringUtils.defaultString;


/**
 * Модель изменения из бинлога.
 */
@ParametersAreNonnullByDefault
public class BinlogEvent {
    private static final List EMPTY_LIST = Collections.emptyList();

    private LocalDateTime utcTimestamp;

    private String source;

    private String serverUuid;

    private long transactionId = -1;

    private int queryIndex = -1;

    // не сериализуется в protobuf
    private int eventIndex = -1;

    @JsonIgnore
    // из пачки событий на запрос для одного надо отправить в Logbroker сам запрос. Это поле помечает такие события
    private boolean writeQuery = false;

    private String db;

    private String table;

    private Operation operation;

    private long traceInfoReqId = 0;

    private String traceInfoService = "";

    private String traceInfoMethod = "";

    private long traceInfoOperatorUid = 0;

    // не сериализуется в protobuf/json, пишется в отдельный топик
    @JsonIgnore
    private String query;

    private String essTag = "";

    private boolean resharding = false;

    @Nonnull
    @SuppressWarnings("unchecked")
    private List<Row> rows = EMPTY_LIST;

    @Nonnull
    @SuppressWarnings("unchecked")
    private List<SchemaChange> schemaChanges = EMPTY_LIST;

    public static BinlogEvent fromProtobuf(BinlogEventProtobuf.Event event) {
        BinlogEvent result = new BinlogEvent();
        result.setUtcTimestamp(LocalDateTime.ofInstant(Instant.ofEpochSecond(event.getTimestamp()), ZoneId.of("UTC")));
        result.setSource(event.getSource());
        result.setServerUuid(event.getServerUuid());
        result.setTransactionId(event.getGtidTransactionId());
        result.setQueryIndex(event.getQueryIndex());
        result.setDb(event.getDb());
        result.setTable(event.getTable());
        result.setTraceInfoReqId(event.getTranceInfoReqId());
        result.setTraceInfoService(event.getTranceInfoService());
        result.setTraceInfoMethod(event.getTranceInfoMethod());
        result.setTraceInfoOperatorUid(event.getTranceInfoOperatorUid());
        result.setEssTag(event.getEssTag());
        result.setResharding(event.getResharding());
        try {
            result.setOperation(Operation.valueOf(event.getOperation().name()));
        } catch (IllegalArgumentException | NullPointerException ignored) {
            throw new IllegalStateException(String.format("Got unexpected operation = %s", result.getOperation()));
        }
        for (BinlogEventProtobuf.Row protoRow : event.getRowsList()) {
            result.addRows(Row.fromProtobuf(protoRow));
        }
        for (SchemaProtobuf.SchemaChange protoSchemaChange : event.getSchemaChangesList()) {
            result.addSchemaChanges(SchemaChange.fromProtobuf(protoSchemaChange));
        }
        return result;
    }

    /**
     * Создает новый инстансе по указанному шаблону. Копирует все поля из шаблона за исключением rows и schemaChanges.
     * Важно: входной шаблон не обязан быть валидным, результат тоже может быть не валидным.
     */
    public static BinlogEvent fromTemplate(BinlogEvent template) {
        return new BinlogEvent()
                .withSource(template.getSource())
                .withServerUuid(template.getServerUuid())
                .withDb(template.getDb())
                .withTable(template.getTable())
                .withTransactionId(template.getTransactionId())
                .withQuery(template.getQuery())
                .withQueryIndex(template.getQueryIndex())
                .withEventIndex(template.getEventIndex())
                .withWriteQuery(template.getWriteQuery())
                .withOperation(template.getOperation())
                .withUtcTimestamp(template.getUtcTimestamp())
                .withTraceInfoReqId(template.getTraceInfoReqId())
                .withTraceInfoService(template.getTraceInfoService())
                .withTraceInfoMethod(template.getTraceInfoMethod())
                .withTraceInfoOperatorUid(template.getTraceInfoOperatorUid())
                .withEssTag(template.getEssTag())
                .withResharding(template.isResharding());

        // validate не вызываем, потому что возвращаем полуфабрикат,
        // некоторые поля в template могут быть пустыми
    }


    /**
     * @return Дата и время события в UTC.
     */
    public LocalDateTime getUtcTimestamp() {
        return utcTimestamp;
    }

    /**
     * @param utcTimestamp Установить дату и время события в UTC.
     */
    public void setUtcTimestamp(LocalDateTime utcTimestamp) {
        this.utcTimestamp = utcTimestamp;
    }

    /**
     * @param utcTimestamp Установить дату и время события в UTC.
     */
    public BinlogEvent withUtcTimestamp(LocalDateTime utcTimestamp) {
        setUtcTimestamp(utcTimestamp);
        return this;
    }

    public String getSource() {
        return source;
    }

    public void setSource(String source) {
        this.source = source;
    }

    public BinlogEvent withSource(String source) {
        setSource(source);
        return this;
    }

    public String getServerUuid() {
        return serverUuid;
    }

    public void setServerUuid(String serverUuid) {
        this.serverUuid = serverUuid;
    }

    public BinlogEvent withServerUuid(String serverUuid) {
        setServerUuid(serverUuid);
        return this;
    }

    public long getTransactionId() {
        return transactionId;
    }

    public void setTransactionId(long transactionId) {
        this.transactionId = transactionId;
    }

    public BinlogEvent withTransactionId(long transactionId) {
        setTransactionId(transactionId);
        return this;
    }

    public String getGtid() {
        return getServerUuid() + ":" + getTransactionId();
    }

    public void setGtid(String gtid) {
        String[] parsedGtid = gtid.split(":");
        setServerUuid(parsedGtid[0]);
        setTransactionId(Long.parseLong(parsedGtid[1]));
    }

    public BinlogEvent withGtid(String gtid) {
        setGtid(gtid);
        return this;
    }

    public int getQueryIndex() {
        return queryIndex;
    }

    public void setQueryIndex(int queryIndex) {
        this.queryIndex = queryIndex;
    }

    public BinlogEvent withQueryIndex(int queryIndex) {
        setQueryIndex(queryIndex);
        return this;
    }

    public int getEventIndex() {
        return eventIndex;
    }

    public void setEventIndex(int eventIndex) {
        this.eventIndex = eventIndex;
    }

    public BinlogEvent withEventIndex(int eventIndex) {
        setEventIndex(eventIndex);
        return this;
    }

    public boolean getWriteQuery() {
        return writeQuery;
    }

    public void setWriteQuery(boolean writeQuery) {
        this.writeQuery = writeQuery;
    }

    public BinlogEvent withWriteQuery(boolean writeQuery) {
        setWriteQuery(writeQuery);
        return this;
    }

    public String getDb() {
        return db;
    }

    public void setDb(String db) {
        this.db = db;
    }

    public BinlogEvent withDb(String db) {
        setDb(db);
        return this;
    }

    public String getTable() {
        return table;
    }

    public void setTable(String table) {
        this.table = table;
    }

    public BinlogEvent withTable(String table) {
        setTable(table);
        return this;
    }

    public Operation getOperation() {
        return operation;
    }

    public void setOperation(Operation operation) {
        this.operation = operation;
    }

    public BinlogEvent withOperation(Operation operation) {
        setOperation(operation);
        return this;
    }

    public long getTraceInfoReqId() {
        return traceInfoReqId;
    }

    public void setTraceInfoReqId(long traceInfoReqId) {
        this.traceInfoReqId = traceInfoReqId;
    }

    public BinlogEvent withTraceInfoReqId(long traceInfoReqId) {
        setTraceInfoReqId(traceInfoReqId);
        return this;
    }

    public String getTraceInfoService() {
        return traceInfoService;
    }

    public void setTraceInfoService(String traceInfoService) {
        this.traceInfoService = traceInfoService;
    }

    public BinlogEvent withTraceInfoService(String traceInfoService) {
        setTraceInfoService(traceInfoService);
        return this;
    }

    public String getTraceInfoMethod() {
        return traceInfoMethod;
    }

    public void setTraceInfoMethod(String traceInfoMethod) {
        this.traceInfoMethod = traceInfoMethod;
    }

    public BinlogEvent withTraceInfoMethod(String traceInfoMethod) {
        setTraceInfoMethod(traceInfoMethod);
        return this;
    }

    public long getTraceInfoOperatorUid() {
        return traceInfoOperatorUid;
    }

    public void setTraceInfoOperatorUid(long traceInfoOperatorUid) {
        this.traceInfoOperatorUid = traceInfoOperatorUid;
    }

    public BinlogEvent withTraceInfoOperatorUid(long traceInfoOperatorUid) {
        setTraceInfoOperatorUid(traceInfoOperatorUid);
        return this;
    }

    public String getQuery() {
        return query;
    }

    public void setQuery(String query) {
        this.query = query;
    }

    public BinlogEvent withQuery(String query) {
        setQuery(query);
        return this;
    }

    public String getEssTag() {
        return essTag;
    }

    public void setEssTag(String essTag) {
        this.essTag = essTag;
    }

    public BinlogEvent withEssTag(String essTag) {
        setEssTag(essTag);
        return this;
    }

    public boolean isResharding() {
        return resharding;
    }

    public BinlogEvent setResharding(boolean resharding) {
        this.resharding = resharding;
        return this;
    }

    public BinlogEvent withResharding(boolean resharding) {
        setResharding(resharding);
        return this;
    }

    @Override
    public String toString() {
        return "BinlogEvent{" +
                "utcTimestamp=" + utcTimestamp +
                ", source='" + source + '\'' +
                ", serverUuid='" + serverUuid + '\'' +
                ", transactionId=" + transactionId +
                ", queryIndex=" + queryIndex +
                ", eventIndex=" + eventIndex +
                ", writeQuery=" + writeQuery +
                ", db='" + db + '\'' +
                ", table='" + table + '\'' +
                ", operation=" + operation +
                ", traceInfoReqId=" + traceInfoReqId +
                ", traceInfoService='" + traceInfoService + '\'' +
                ", traceInfoMethod='" + traceInfoMethod + '\'' +
                ", traceInfoOperatorUid=" + traceInfoOperatorUid +
                ", essTag='" + essTag + '\'' +
                ", resharding=" + resharding +
                ", rows=" + rows +
                ", schemaChanges=" + schemaChanges +
                '}';
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        BinlogEvent that = (BinlogEvent) o;
        return transactionId == that.transactionId &&
                queryIndex == that.queryIndex &&
                eventIndex == that.eventIndex &&
                writeQuery == that.writeQuery &&
                traceInfoReqId == that.traceInfoReqId &&
                traceInfoOperatorUid == that.traceInfoOperatorUid &&
                resharding == that.resharding &&
                Objects.equals(utcTimestamp, that.utcTimestamp) &&
                Objects.equals(source, that.source) &&
                Objects.equals(serverUuid, that.serverUuid) &&
                Objects.equals(db, that.db) &&
                Objects.equals(table, that.table) &&
                operation == that.operation &&
                Objects.equals(traceInfoService, that.traceInfoService) &&
                Objects.equals(traceInfoMethod, that.traceInfoMethod) &&
                Objects.equals(essTag, that.essTag) &&
                rows.equals(that.rows) &&
                schemaChanges.equals(that.schemaChanges);
    }

    @Override
    public int hashCode() {
        return Objects.hash(utcTimestamp, source, serverUuid, transactionId, queryIndex, eventIndex,
                writeQuery, db, table, operation, traceInfoReqId, traceInfoService, traceInfoMethod,
                traceInfoOperatorUid, essTag, resharding, rows, schemaChanges);
    }

    public BinlogEvent validate() {
        Objects.requireNonNull(getDb(), "db should not be null");
        Objects.requireNonNull(getOperation(), "operation should not be null");
        Objects.requireNonNull(getServerUuid(), "serverUuid should not be null");
        Objects.requireNonNull(getSource(), "source should not be null");
        Objects.requireNonNull(getTable(), "table should not be null");
        Objects.requireNonNull(getUtcTimestamp(), "utcTimestamp should not be null");

        Preconditions.checkState(getQueryIndex() >= 0, "queryIndex should not be null");
        Preconditions.checkState(getTransactionId() >= 0, "transactionId should not be null");

        String emptinessErrorPattern = "When operation = %s %s should%s be empty";
        boolean rowsShouldBeEmpty = getOperation() == Operation.SCHEMA;
        boolean schemaShouldBeEmpty = getOperation() != Operation.SCHEMA;

        if (getRows().isEmpty() != rowsShouldBeEmpty) {
            throw new IllegalStateException(String.format(emptinessErrorPattern, getOperation(), "rows",
                    rowsShouldBeEmpty ? "" : " not"));
        }
        if (getSchemaChanges().isEmpty() != schemaShouldBeEmpty) {
            throw new IllegalStateException(String.format(emptinessErrorPattern, getOperation(), "schemaChanges",
                    schemaShouldBeEmpty ? "" : " not"));
        }

        for (int index = 0; index < rows.size(); ++index) {
            try {
                rows.get(index).validate();
            } catch (RuntimeException e) {
                throw new IllegalStateException("rows[" + index + "]: " + e.getMessage(), e);
            }
        }

        for (int index = 0; index < schemaChanges.size(); ++index) {
            try {
                schemaChanges.get(index).validate();
            } catch (RuntimeException e) {
                throw new IllegalStateException("schemaChanges[" + index + "]: " + e.getMessage(), e);
            }
        }

        return this;
    }

    public BinlogEventProtobuf.Event toProtobuf() {
        BinlogEventProtobuf.Event.Builder builder = BinlogEventProtobuf.Event.newBuilder()
                .setTimestamp(getUtcTimestampAsEpochSecond())
                .setSource(getSource())
                .setServerUuid(getServerUuid())
                .setGtidTransactionId(getTransactionId())
                .setQueryIndex(getQueryIndex())
                .setDb(getDb())
                .setTable(getTable())
                .setQueryChunkIndex(getQueryChunkIndex())
                .setTranceInfoReqId(getTraceInfoReqId())
                .setTranceInfoOperatorUid(getTraceInfoOperatorUid())
                .setTranceInfoService(defaultString(getTraceInfoService()))
                .setTranceInfoMethod(defaultString(getTraceInfoMethod()))
                .setEssTag(defaultString(getEssTag()))
                .setResharding(isResharding());

        switch (Objects.requireNonNull(getOperation())) {
            case INSERT:
                builder.setOperation(BinlogEventProtobuf.Operation.INSERT);
                break;
            case UPDATE:
                builder.setOperation(BinlogEventProtobuf.Operation.UPDATE);
                break;
            case DELETE:
                builder.setOperation(BinlogEventProtobuf.Operation.DELETE);
                break;
            case SCHEMA:
                builder.setOperation(BinlogEventProtobuf.Operation.SCHEMA);
                break;
            default:
                throw new IllegalArgumentException(getOperation().toString());
        }
        for (Row row : getRows()) {
            builder.addRows(row.toProtobuf());
        }
        for (SchemaChange schemaChange : getSchemaChanges()) {
            builder.addSchemaChanges(schemaChange.toProtobuf());
        }
        return builder.build();
    }

    @JsonGetter("queryChunkIndex")
    public Integer getQueryChunkIndex() {
        return rows.stream()
                .map(Row::getRowIndex)
                .min(Comparator.naturalOrder())
                .orElse(0);
    }

    @JsonGetter("utcTimestamp")
    public long getUtcTimestampAsEpochSecond() {
        return getUtcTimestamp().toEpochSecond(ZoneOffset.UTC);
    }

    public void addRows(Row... rows) {
        if (getRows() == EMPTY_LIST) {
            setRows(new ArrayList<>());
        }
        getRows().addAll(Arrays.asList(rows));
    }

    public BinlogEvent withAddedRows(Row... rows) {
        addRows(rows);
        return this;
    }

    public BinlogEvent withRows(List<Row> rows) {
        setRows(rows);
        return this;
    }

    public List<Row> getRows() {
        return rows;
    }

    public void setRows(List<Row> rows) {
        this.rows = rows;
    }

    public void addSchemaChanges(SchemaChange... schemaChanges) {
        if (getSchemaChanges() == EMPTY_LIST) {
            setSchemaChanges(new ArrayList<>());
        }
        getSchemaChanges().addAll(Arrays.asList(schemaChanges));
    }

    public BinlogEvent withAddedSchemaChanges(SchemaChange... schemaChanges) {
        addSchemaChanges(schemaChanges);
        return this;
    }

    public BinlogEvent withSchemaChanges(List<SchemaChange> schemaChanges) {
        setSchemaChanges(schemaChanges);
        return this;
    }

    public List<SchemaChange> getSchemaChanges() {
        return schemaChanges;
    }

    public void setSchemaChanges(List<SchemaChange> schemaChanges) {
        this.schemaChanges = schemaChanges;
    }

    public static class Row {
        private static final ProtobufValueConverter PROTOBUF_CONVERTER = new ProtobufValueConverter();

        private int rowIndex = -1;

        private Map<String, Object> primaryKey;

        private Map<String, Object> before;

        private Map<String, Object> after;

        public static Row fromProtobuf(BinlogEventProtobuf.Row protoRow) {
            Row result = new Row();
            result.setRowIndex(protoRow.getRowIndex());
            result.setPrimaryKey(toMap(protoRow.getPrimaryKeyList()));
            result.setBefore(toMap(protoRow.getBeforeList()));
            result.setAfter(toMap(protoRow.getAfterList()));
            return result;
        }

        private static Map<String, Object> toMap(
                Collection<BinlogEventProtobuf.ColumnValue> columnValues) {
            Map<String, Object> result = new HashMap<>();
            for (BinlogEventProtobuf.ColumnValue columnValue : columnValues) {
                result.put(columnValue.getName(), PROTOBUF_CONVERTER.doBackward(columnValue.getValue()));
            }
            return result;
        }

        private static void fillColumnValues(Consumer<BinlogEventProtobuf.ColumnValue> consumer,
                                             Collection<Map.Entry<String, Object>> collection) {
            for (Map.Entry<String, Object> entry : collection) {
                consumer.accept(BinlogEventProtobuf.ColumnValue.newBuilder()
                        .setName(entry.getKey())
                        .setValue(PROTOBUF_CONVERTER.doForward(entry.getValue()))
                        .build());
            }
        }

        public int getRowIndex() {
            return rowIndex;
        }

        public void setRowIndex(int rowIndex) {
            this.rowIndex = rowIndex;
        }

        public Row withRowIndex(int rowIndex) {
            setRowIndex(rowIndex);
            return this;
        }

        public Map<String, Object> getPrimaryKey() {
            return primaryKey;
        }

        public void setPrimaryKey(Map<String, Object> primaryKey) {
            this.primaryKey = primaryKey;
        }

        public Row withPrimaryKey(Map<String, Object> primaryKey) {
            setPrimaryKey(primaryKey);
            return this;
        }

        public Map<String, Object> getBefore() {
            return before;
        }

        public void setBefore(Map<String, Object> before) {
            this.before = before;
        }

        public Row withBefore(Map<String, Object> before) {
            setBefore(before);
            return this;
        }

        public Map<String, Object> getAfter() {
            return after;
        }

        public void setAfter(Map<String, Object> after) {
            this.after = after;
        }

        public Row withAfter(Map<String, Object> after) {
            setAfter(after);
            return this;
        }

        public Row validate() {
            Objects.requireNonNull(getAfter(), "after should not be null");
            Objects.requireNonNull(getBefore(), "before should not be null");
            Objects.requireNonNull(getPrimaryKey(), "primaryKey should not be null");
            // before может быть пустым в любом случае, after может быть пустым при удалении
            // DIRECT-81764 Раньше была проверка обязательного наличия primary key, но в Директе
            // оказалось немного таблиц без primary key.
            Preconditions.checkState(getRowIndex() >= 0, "rowIndex should not be null");
            return this;
        }

        public BinlogEventProtobuf.Row toProtobuf() {
            BinlogEventProtobuf.Row.Builder builder = BinlogEventProtobuf.Row.newBuilder()
                    .setRowIndex(getRowIndex());
            fillColumnValues(builder::addPrimaryKey, Objects.requireNonNull(getPrimaryKey()).entrySet());
            fillColumnValues(builder::addBefore, Objects.requireNonNull(getBefore()).entrySet());
            fillColumnValues(builder::addAfter, Objects.requireNonNull(getAfter()).entrySet());
            return builder.build();
        }

        @Override
        public String toString() {
            return "Row{" +
                    "rowIndex=" + rowIndex +
                    ", primaryKey=" + primaryKey +
                    ", before=" + before +
                    ", after=" + after +
                    '}';
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (!(o instanceof Row)) {
                return false;
            }
            Row row = (Row) o;
            return rowIndex == row.rowIndex &&
                    Objects.equals(primaryKey, row.primaryKey) &&
                    Objects.equals(before, row.before) &&
                    Objects.equals(after, row.after);
        }

        @Override
        public int hashCode() {
            return Objects.hash(rowIndex, primaryKey, before, after);
        }
    }
}
