package ru.yandex.direct.binlog.model;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.util.BitSet;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import static ru.yandex.direct.utils.DateTimeUtils.MSK;

/**
 * Тип для колонки. Отбрасывает детали реализации, нужные для хранилища (например, varchar и text - это {@link #STRING}.
 */
@ParametersAreNonnullByDefault
public enum ColumnType {
    /**
     * Для хранения <pre>byte[]</pre>
     */
    BYTES,

    /**
     * Для хранения {@link java.time.LocalDate LocalDate}
     */
    DATE,

    /**
     * Для хранения {@link java.math.BigDecimal BigDecimal}
     */
    FIXED_POINT,

    /**
     * Для хранения double
     */
    FLOATING_POINT,

    /**
     * Для хранения long
     */
    INTEGER,

    /**
     * Для хранения {@link String}
     */
    STRING,

    /**
     * Для хранения {@link java.time.LocalDateTime LocalDateTime}
     */
    TIMESTAMP,

    /**
     * Для хранения {@link java.math.BigInteger BigInteger}
     */
    UNSIGNED_BIGINT;

    /**
     * Все значения LocalDateTime хранятся в этой зоне
     */
    public static final ZoneId TIMESTAMP_ZONE_ID = MSK;

    /**
     * Приводит объект к некоему общему типу, который по смыслу равен исходному значению. Например, превращает int в
     * long, но ни во что не превращает String.
     *
     * @param object Исходный объект.
     * @return null, если исходный объект тоже null. Иначе пара - тип объекта и значение, приведённое к общему типу.
     */
    @Nullable
    public static Normalized normalize(@Nullable Object object) {
        if (object == null) {
            return null;
        } else if (object instanceof byte[]) {
            return new Normalized(BYTES, object);
        } else if (object instanceof LocalDate) {
            return new Normalized(DATE, object);
        } else if (object instanceof java.sql.Date) {
            return new Normalized(DATE, ((java.sql.Date) object).toLocalDate());
        } else if (object instanceof BigDecimal) {
            return new Normalized(FIXED_POINT, object);
        } else if (object instanceof Double) {
            return new Normalized(FLOATING_POINT, object);
        } else if (object instanceof Float) {
            return new Normalized(FLOATING_POINT, (double) (float) object);
        } else if (object instanceof Long) {
            return new Normalized(INTEGER, object);
        } else if (object instanceof Integer) {
            return new Normalized(INTEGER, (long) (int) object);
        } else if (object instanceof Short) {
            return new Normalized(INTEGER, (long) (short) object);
        } else if (object instanceof Byte) {
            return new Normalized(INTEGER, (long) (byte) object);
        } else if (object instanceof Boolean) {
            return new Normalized(INTEGER, ((Boolean) object) ? 1L : 0L);
        } else if (object instanceof String) {
            return new Normalized(STRING, object);
        } else if (object instanceof BitSet) {
            return new Normalized(BYTES, ((BitSet) object).toByteArray());
        } else if (object instanceof LocalDateTime) {
            return new Normalized(TIMESTAMP, object);
        } else if (object instanceof java.util.Date && !(object instanceof java.sql.Time)) {
            // java.sql.Timestamp наследуется от java.util.Date.
            // При этом java.util.Date и java.sql.Date означают разные вещи.
            // Для java.sql.Time, который представляет собой только время без даты, обработка пока не предусмотрена.
            return new Normalized(TIMESTAMP,
                    LocalDateTime.ofInstant(((java.util.Date) object).toInstant(), TIMESTAMP_ZONE_ID));
        } else if (object instanceof BigInteger) {
            return new Normalized(UNSIGNED_BIGINT, object);
        } else {
            throw new IllegalArgumentException("Don't know how to normalize `" + object
                    + "` with type " + object.getClass().getCanonicalName());
        }
    }

    public static class Normalized {
        private final ColumnType type;
        private final Object object;

        private Normalized(ColumnType type, Object object) {
            this.type = type;
            this.object = object;
        }

        public ColumnType getType() {
            return type;
        }

        public Object getObject() {
            return object;
        }
    }
}
