package ru.yandex.direct.binlog.model;

import java.util.Objects;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Preconditions;
import org.apache.commons.lang3.StringUtils;

/**
 * Событие создания или изменения колонки.
 * Если колонки не существовало, она должна быть создана. Если колонка существовала, у неё должен быть измененён тип.
 * Название таблицы - в {@link BinlogEvent#getTable()}.
 */
@ParametersAreNonnullByDefault
public class CreateOrModifyColumn implements SchemaChange<CreateOrModifyColumn> {
    private String columnName;
    private ColumnType columnType;
    private boolean nullable;
    @Nullable
    private String defaultValue;

    public String getColumnName() {
        return columnName;
    }

    public void setColumnName(String columnName) {
        this.columnName = columnName;
    }

    public CreateOrModifyColumn withColumnName(String columnName) {
        setColumnName(columnName);
        return this;
    }

    @Nonnull
    public ColumnType getColumnType() {
        return columnType;
    }

    public void setColumnType(@Nonnull ColumnType columnType) {
        this.columnType = columnType;
    }

    public CreateOrModifyColumn withColumnType(@Nonnull ColumnType columnType) {
        setColumnType(columnType);
        return this;
    }

    public boolean isNullable() {
        return nullable;
    }

    public void setNullable(boolean nullable) {
        this.nullable = nullable;
    }

    public CreateOrModifyColumn withNullable(boolean nullable) {
        setNullable(nullable);
        return this;
    }

    @Nullable
    public String getDefaultValue() {
        return defaultValue;
    }

    public void setDefaultValue(@Nullable String defaultValue) {
        this.defaultValue = defaultValue;
    }

    public CreateOrModifyColumn withDefaultValue(@Nullable String defaultValue) {
        setDefaultValue(defaultValue);
        return this;
    }

    public CreateOrModifyColumn validate() {
        Preconditions.checkState(!StringUtils.isEmpty(columnName), "columnName is null or empty");
        Objects.requireNonNull(columnType, "columnType is null");
        return this;
    }

    public SchemaProtobuf.SchemaChange toProtobuf() {
        return SchemaProtobuf.SchemaChange.newBuilder()
                .setCreateOrModifyColumn(toProtobufDirect())
                .build();
    }

    @Nonnull
    SchemaProtobuf.CreateOrModifyColumn toProtobufDirect() {
        return SchemaProtobuf.CreateOrModifyColumn.newBuilder()
                .setColumName(getColumnName())
                .setType(SchemaProtobuf.ColumnType.valueOf(columnType.toString()))
                .setNullable(isNullable())
                .setDefaultValue(defaultValue == null ? "" : defaultValue)
                .setHasDefaultValue(defaultValue != null)
                .build();
    }

    public static CreateOrModifyColumn fromProtobuf(SchemaProtobuf.SchemaChange protoSchemaChange) {
        return fromProtobuf(protoSchemaChange.getCreateOrModifyColumn());
    }

    public static CreateOrModifyColumn fromProtobuf(SchemaProtobuf.CreateOrModifyColumn protoCreateOrModifyColumn) {
        return new CreateOrModifyColumn()
                .withColumnName(protoCreateOrModifyColumn.getColumName())
                .withColumnType(ColumnType.valueOf(protoCreateOrModifyColumn.getType().name()))
                .withNullable(protoCreateOrModifyColumn.getNullable())
                .withDefaultValue(protoCreateOrModifyColumn.getHasDefaultValue() ?
                        protoCreateOrModifyColumn.getDefaultValue() : null);
    }

    @Override
    public String toString() {
        return "CreateOrModifyColumn{" +
                "columnName='" + columnName + '\'' +
                ", columnType=" + columnType +
                ", nullable=" + nullable +
                ", defaultValue='" + defaultValue + '\'' +
                '}';
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        CreateOrModifyColumn that = (CreateOrModifyColumn) o;
        return nullable == that.nullable &&
                Objects.equals(columnName, that.columnName) &&
                columnType == that.columnType &&
                Objects.equals(defaultValue, that.defaultValue);
    }

    @Override
    public int hashCode() {
        return Objects.hash(columnName, columnType, nullable, defaultValue);
    }
}
