package ru.yandex.direct.binlog.model;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;

import javax.annotation.ParametersAreNonnullByDefault;

/**
 * Создание новой таблицы с рядом колонок. Название таблицы - в {@link BinlogEvent#getTable()}.
 * Содержит описание колонок через {@link CreateOrModifyColumn}, а также названия колонок,
 * из которых составлен первичный ключ (если есть).
 */
@ParametersAreNonnullByDefault
public class CreateTable implements SchemaChange<CreateTable> {
    private List<CreateOrModifyColumn> columns = new ArrayList<>();
    private List<String> primaryKey;

    public static CreateTable fromProtobuf(SchemaProtobuf.SchemaChange protoSchemaChange) {
        var ret = new CreateTable();
        protoSchemaChange.getCreateTable().getColumnsList().forEach(
                c -> ret.columns.add(CreateOrModifyColumn.fromProtobuf(c)));
        ret.primaryKey = new ArrayList<>(protoSchemaChange.getCreateTable().getPrimaryKeyList());
        return ret;
    }

    @Override
    public CreateTable validate() {
        return this;
    }

    @Override
    public SchemaProtobuf.SchemaChange toProtobuf() {
        SchemaProtobuf.CreateTable.Builder createTable = SchemaProtobuf.CreateTable.newBuilder();
        columns.forEach(c -> createTable.addColumns(c.toProtobufDirect()));
        primaryKey.forEach(createTable::addPrimaryKey);
        return SchemaProtobuf.SchemaChange.newBuilder()
                .setCreateTable(createTable)
                .build();
    }

    public List<String> getPrimaryKey() {
        return primaryKey;
    }

    public void setPrimaryKey(List<String> primaryKey) {
        this.primaryKey = primaryKey;
    }

    public CreateTable withPrimaryKey(List<String> primaryKey) {
        setPrimaryKey(primaryKey);
        return this;
    }

    public List<CreateOrModifyColumn> getColumns() {
        return columns;
    }

    public void setColumns(List<CreateOrModifyColumn> columns) {
        this.columns = columns;
    }

    public CreateTable withColumns(List<CreateOrModifyColumn> columns) {
        setColumns(columns);
        return this;
    }

    public void addColumns(CreateOrModifyColumn... columns) {
        this.columns.addAll(Arrays.asList(columns));
    }

    public CreateTable withAddedColumns(CreateOrModifyColumn... columns) {
        addColumns(columns);
        return this;
    }

    @Override
    public String toString() {
        return "CreateTable{" +
                "columns=" + columns +
                ", primaryKey=" + primaryKey +
                '}';
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (!(o instanceof CreateTable)) {
            return false;
        }
        CreateTable that = (CreateTable) o;
        return Objects.equals(columns, that.columns) &&
                Objects.equals(primaryKey, that.primaryKey);
    }

    @Override
    public int hashCode() {
        return Objects.hash(columns, primaryKey);
    }
}
