package ru.yandex.direct.binlog.model;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.ZoneOffset;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Converter;
import com.google.protobuf.ByteString;
import com.google.protobuf.NullValue;

@ParametersAreNonnullByDefault
class ProtobufValueConverter extends Converter<Object, TypesProtobuf.Value> {
    private static BigInteger fromUint64(long value) {
        BigInteger result = BigInteger.valueOf(value & Long.MAX_VALUE);
        if ((value & Long.MIN_VALUE) != 0) {
            return result.shiftLeft(1).add(BigInteger.ONE);
        } else {
            return result;
        }
    }

    private static long toUint64(BigInteger value) {
        return value.longValue();
    }

    @Override
    protected TypesProtobuf.Value doForward(@Nullable Object o) {
        TypesProtobuf.Value.Builder builder = TypesProtobuf.Value.newBuilder();
        ColumnType.Normalized normalized = ColumnType.normalize(o);
        if (normalized == null) {
            builder.setNull(NullValue.NULL_VALUE);
        } else {
            switch (normalized.getType()) {
                case BYTES:
                    builder.setBytes(ByteString.copyFrom((byte[]) normalized.getObject()));
                    break;
                case DATE:
                    builder.setDate(((LocalDate) normalized.getObject()).toEpochDay());
                    break;
                case FIXED_POINT:
                    builder.setFixedPoint(((BigDecimal) normalized.getObject()).toPlainString());
                    break;
                case FLOATING_POINT:
                    builder.setFloatingPoint((Double) normalized.getObject());
                    break;
                case INTEGER:
                    builder.setInteger((Long) normalized.getObject());
                    break;
                case STRING:
                    builder.setString((String) normalized.getObject());
                    break;
                case TIMESTAMP:
                    LocalDateTime value = (LocalDateTime) normalized.getObject();
                    builder.setTimestamp(com.google.protobuf.Timestamp.newBuilder()
                            .setSeconds(value.toEpochSecond(ZoneOffset.UTC))
                            .setNanos(value.getNano()));
                    break;
                case UNSIGNED_BIGINT:
                    builder.setUnsignedBigint(toUint64((BigInteger) normalized.getObject()));
                    break;
                default:
                    throw new IllegalArgumentException("Don't know how to encode `" + o + "` with type "
                            + (o == null ? null : o.getClass().getCanonicalName()));
            }
        }
        return builder.build();
    }

    @Nullable
    @Override
    protected Object doBackward(TypesProtobuf.Value value) {
        // В этом горячем месте лучше не использовать Preconditions, чтобы не приходилось вычислять
        // value.toString() -- иначе это будет занимать до 30% времени всей десериализации
        if (value.getValueCase() == null) {
            throw new IllegalArgumentException("Don't know how to decode `" + value + "` without defined type");
        }
        switch (value.getValueCase()) {
            case BYTES:
                return value.getBytes().toByteArray();
            case DATE:
                return LocalDate.ofEpochDay(value.getDate());
            case FIXED_POINT:
                return new BigDecimal(value.getFixedPoint());
            case FLOATING_POINT:
                return value.getFloatingPoint();
            case INTEGER:
                return value.getInteger();
            case NULL:
                return null;
            case STRING:
                return value.getString();
            case TIMESTAMP:
                return LocalDateTime.ofEpochSecond(value.getTimestamp().getSeconds(),
                        value.getTimestamp().getNanos(),
                        ZoneOffset.UTC);
            case UNSIGNED_BIGINT:
                return fromUint64(value.getUnsignedBigint());
            default:
                throw new IllegalArgumentException(
                        "Don't know how to decode `" + value + "` with case " + value.getValueCase());
        }
    }
}
