package ru.yandex.direct.binlog.model;

import java.util.Objects;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Preconditions;
import org.apache.commons.lang3.StringUtils;

/**
 * Событие переименования колонки в таблице. Название таблицы - в {@link BinlogEvent#getTable()}.
 */
@ParametersAreNonnullByDefault
public class RenameColumn implements SchemaChange {
    private String oldColumnName;
    private String newColumnName;

    public String getOldColumnName() {
        return oldColumnName;
    }

    public RenameColumn withOldColumnName(String oldColumnName) {
        this.oldColumnName = oldColumnName;
        return this;
    }

    public String getNewColumnName() {
        return newColumnName;
    }

    public RenameColumn withNewColumnName(String newColumnName) {
        this.newColumnName = newColumnName;
        return this;
    }

    @Override
    public SchemaChange validate() {
        Preconditions.checkState(!StringUtils.isEmpty(oldColumnName), "oldColumnName is null or empty");
        Preconditions.checkState(!StringUtils.isEmpty(newColumnName), "newColumnName is null or empty");
        return this;
    }

    @Override
    public SchemaProtobuf.SchemaChange toProtobuf() {
        return SchemaProtobuf.SchemaChange.newBuilder()
                .setRenameColumn(SchemaProtobuf.RenameColumn.newBuilder()
                        .setOldColumnName(getOldColumnName())
                        .setNewColumnName(getNewColumnName()))
                .build();
    }

    public static RenameColumn fromProtobuf(SchemaProtobuf.SchemaChange protoSchemaChange) {
        SchemaProtobuf.RenameColumn renameColumn = protoSchemaChange.getRenameColumn();
        return new RenameColumn().withOldColumnName(renameColumn.getOldColumnName())
                .withNewColumnName(renameColumn.getNewColumnName());
    }

    @Override
    public String toString() {
        return "RenameColumn{" +
                "oldColumnName='" + oldColumnName + '\'' +
                ", newColumnName='" + newColumnName + '\'' +
                '}';
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (!(o instanceof RenameColumn)) {
            return false;
        }
        RenameColumn that = (RenameColumn) o;
        return Objects.equals(oldColumnName, that.oldColumnName) &&
                Objects.equals(newColumnName, that.newColumnName);
    }

    @Override
    public int hashCode() {
        return Objects.hash(oldColumnName, newColumnName);
    }
}
