package ru.yandex.direct.binlog.model;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Preconditions;

@ParametersAreNonnullByDefault
public class RenameTable implements SchemaChange<RenameTable> {
    private List<RenameSingleTable> renames = new ArrayList<>();

    public List<RenameSingleTable> getRenames() {
        return renames;
    }

    public RenameTable withRenames(List<RenameSingleTable> renames) {
        this.renames = renames;
        return this;
    }

    public RenameTable withAddRename(RenameSingleTable rename) {
        this.renames.add(rename);
        return this;
    }

    public RenameTable withAddRename(String oldTableName, String newTableName) {
        return withAddRename(new RenameSingleTable(oldTableName, newTableName));
    }

    public static RenameTable fromProtobuf(SchemaProtobuf.SchemaChange protoSchemaChange) {
        var rename = new RenameTable();
        SchemaProtobuf.RenameTable renameTable = protoSchemaChange.getRenameTable();
        for (SchemaProtobuf.RenameSingleTable renameSingleTable : renameTable.getRenameList()) {
            rename.withAddRename(renameSingleTable.getOldTableName(), renameSingleTable.getNewTableName());
        }
        return rename;
    }

    @Override
    public RenameTable validate() {
        Preconditions.checkState(!renames.isEmpty(), "renames in empty");
        return this;
    }

    @Override
    public SchemaProtobuf.SchemaChange toProtobuf() {
        SchemaProtobuf.RenameTable.Builder builder = SchemaProtobuf.RenameTable.newBuilder();
        for (RenameSingleTable rename : renames) {
            builder.addRename(SchemaProtobuf.RenameSingleTable.newBuilder()
                    .setNewTableName(rename.getNewTableName())
                    .setOldTableName(rename.getOldTableName()));
        }
        return SchemaProtobuf.SchemaChange.newBuilder()
                .setRenameTable(builder)
                .build();
    }

    @Override
    public String toString() {
        return "RenameTable{" +
                "renames=" + renames +
                '}';
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (!(o instanceof RenameTable)) {
            return false;
        }
        RenameTable that = (RenameTable) o;
        return Objects.equals(renames, that.renames);
    }

    @Override
    public int hashCode() {
        return Objects.hash(renames);
    }

    public static class RenameSingleTable {
        private final String oldTableName;
        private final String newTableName;

        RenameSingleTable(String oldTableName, String newTableName) {
            this.oldTableName = oldTableName;
            this.newTableName = newTableName;
        }

        public String getOldTableName() {
            return oldTableName;
        }

        public String getNewTableName() {
            return newTableName;
        }

        @Override
        public String toString() {
            return "RenameSingleTable{" +
                    "oldTableName='" + oldTableName + '\'' +
                    ", newTableName='" + newTableName + '\'' +
                    '}';
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (!(o instanceof RenameSingleTable)) {
                return false;
            }
            RenameSingleTable that = (RenameSingleTable) o;
            return Objects.equals(oldTableName, that.oldTableName) &&
                    Objects.equals(newTableName, that.newTableName);
        }

        @Override
        public int hashCode() {
            return Objects.hash(oldTableName, newTableName);
        }
    }
}
