package ru.yandex.direct.binlog.reader;

import java.io.EOFException;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.sql.Connection;
import java.sql.SQLException;
import java.time.Duration;
import java.util.ArrayDeque;
import java.util.Queue;
import java.util.concurrent.Semaphore;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;
import javax.annotation.concurrent.NotThreadSafe;

import com.github.shyiko.mysql.binlog.network.ServerException;
import com.google.common.base.Preconditions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.mysql.BinlogDDLException;
import ru.yandex.direct.mysql.BinlogEvent;
import ru.yandex.direct.mysql.BinlogEventConverter;
import ru.yandex.direct.mysql.BinlogRawEventServerSource;
import ru.yandex.direct.mysql.MySQLServerBuilder;
import ru.yandex.direct.mysql.TmpMySQLServerWithDataDir;
import ru.yandex.direct.utils.Checked;
import ru.yandex.direct.utils.MonotonicTime;
import ru.yandex.direct.utils.NanoTimeClock;
import ru.yandex.direct.utils.Transient;
import ru.yandex.direct.utils.db.MySQLConnector;
import ru.yandex.direct.utils.exception.RuntimeTimeoutException;

/**
 * Это утилитарный класс, заведующий инициализацией окружения для работы Reader-а.
 * <p>
 * Основное, что происходит в этом классе: запуск mysql-сервера, на который будет реплицироваться схема
 * сервера-источника; начальная репликация схемы. Обе задачи достаточно тяжелые (в случае БД Директа могут занимать
 * минуту и больше, при этом существенно утилизируют диск).
 * <p>
 * Зачем нужен сервер для репликации схемы: в сыром бинлоге есть только номера колонок и их "загрубленные" типы.
 * В условиях часто-меняющейся схемы нельзя полагаться на номера, меняются и имена колонок и их тип и даже смысл.
 * Такие изменения могут всё испортить, поэтому нужен способ их отслеживать. В принципе, запросы на альтеры таблиц
 * есть в самих бинлогах, но парсить их довольно сложно. Поэтому выбран суровый и надежный способ: каждый альтер
 * из бинлога мы применяем на схеме-реплике и перечитываем информацию об изменившихся таблицах из мускуля в виде,
 * пригодном для ис.пользования.
 * <p>
 * (последний абзац не совсем на своем месте. По хорошему, надо унести его в mysql-streaming)
 */
@NotThreadSafe
@ParametersAreNonnullByDefault
public class BinlogReader implements AutoCloseable {
    private static final Logger logger = LoggerFactory.getLogger(BinlogReader.class);
    @Nullable
    private final TmpMySQLServerWithDataDir schemaReplicaMysql;
    private final Connection schemaReplicaConn;
    private final StatefulBinlogSource source;
    private final BinlogEventConverter converter;
    private final int serverId;
    private final int maxBufferedEvents;
    private final Duration keepAliveTimeout;
    private final Queue<MonotonicTime> exceptionWindow;
    private final Duration exceptionWindowDuration;
    private final int exceptionWindowSize;
    private final Duration reconnectInterval;
    private boolean closed;
    private MonotonicTime lastSeenEventTimestamp;
    @Nullable
    private MonotonicTime lastConnectWasAt;
    private boolean forceReconnect;

    /**
     * @param source                    Источник и его состояние
     * @param schemaReplicaMysqlBuilder builder для запуска mysql-сервера, на котором будет происходит репликация
     *                                  схемы сервера-источника.
     * @param mysqlBuilderSemaphore     Семафор, который позволяет контролировать степень параллелизма при инициализации
     *                                  окружения. Если null, то нет ограничения.
     * @param keepAliveTimeout          Следует переустановить соединение, если не было получено ни одного события
     *                                  за указанный период.
     * @param serverId                  Для чтения бинлога нужно подключится к мастер-серверу как слейв, а для этого нужно представиться,
     *                                  сообщив свой server-id, это такой идентификатор слейва. Для мастера важно, чтобы у всех слейвов
     *                                  были уникальные server-id. Если к мастеру подключится несколько слейвов с одинаковым server-id,
     *                                  то мастер отключит одного из них.
     * @param maxBufferedEvents         Размер буффера для событий из бинлога.
     * @throws InterruptedException
     */
    public BinlogReader(
            StatefulBinlogSource source,
            MySQLServerBuilder schemaReplicaMysqlBuilder,
            @Nullable Semaphore mysqlBuilderSemaphore,
            Duration keepAliveTimeout,
            int serverId,
            int maxBufferedEvents
    )
            throws InterruptedException {
        this(source, schemaReplicaMysqlBuilder, mysqlBuilderSemaphore, keepAliveTimeout, serverId, maxBufferedEvents,
                Duration.ofMinutes(5), 5, Duration.ofSeconds(5));
    }

    /**
     * См. остальные аргументы в {@link #BinlogReader(StatefulBinlogSource, MySQLServerBuilder, Semaphore, Duration, int, int)}.
     *
     * @param exceptionWindowDuration Падать, если произошло несколько ошибок I/O за указанный период.
     * @param exceptionWindowSize     Падать, если за указанный период произошло это количество ошибок I/O.
     * @param reconnectInterval       Минимальное ожидание между попытками установить повторное соединение в случае
     *                                ошибки I/O.
     * @throws InterruptedException
     */
    public BinlogReader(
            StatefulBinlogSource source,
            MySQLServerBuilder schemaReplicaMysqlBuilder,
            @Nullable Semaphore mysqlBuilderSemaphore,
            Duration keepAliveTimeout,
            int serverId,
            int maxBufferedEvents,
            Duration exceptionWindowDuration,
            int exceptionWindowSize,
            Duration reconnectInterval)
            throws InterruptedException {
        Preconditions.checkState(
                // 0.95 взято с потолка, можно заменить на что-то иное, стремящееся к 1 слева.
                reconnectInterval.toMillis() * exceptionWindowSize < 0.95 * exceptionWindowDuration.toMillis(),
                "With exceptionWindowDuration=%s, exceptionWindowSize=%s, reconnectInterval=%s BinlogReader will"
                        + " reconnect endlessly",
                exceptionWindowDuration, exceptionWindowSize, reconnectInterval);
        this.source = source;
        this.keepAliveTimeout = keepAliveTimeout;
        this.serverId = serverId;
        this.maxBufferedEvents = maxBufferedEvents;
        this.exceptionWindowDuration = exceptionWindowDuration;
        this.exceptionWindowSize = exceptionWindowSize;
        this.reconnectInterval = reconnectInterval;
        this.exceptionWindow = new ArrayDeque<>();
        this.converter = new BinlogEventConverter(source.getState());
        try (
                Transient<TmpMySQLServerWithDataDir> mysqlHolder = new Transient<>(makeMysql(mysqlBuilderSemaphore,
                        source.getName(),
                        schemaReplicaMysqlBuilder
                ));
                Transient<Connection> connHolder = new Transient<>(mysqlHolder.item.connect())
        ) {
            // тут происходит восстановление состояния из source.getState()
            this.converter.attachSchemaConnection(connHolder.item);

            this.schemaReplicaConn = connHolder.pop();
            this.schemaReplicaMysql = mysqlHolder.pop();
        } catch (SQLException exc) {
            throw new Checked.CheckedException(exc);
        }
    }

    private static TmpMySQLServerWithDataDir makeMysql(
            @Nullable Semaphore semaphore,
            String sourceName,
            MySQLServerBuilder schemaReplicaMysqlBuilder)
            throws InterruptedException {
        if (semaphore != null) {
            semaphore.acquire();
        }
        try {
            return TmpMySQLServerWithDataDir.create("schema-replica-" + sourceName, schemaReplicaMysqlBuilder);
        } finally {
            if (semaphore != null) {
                semaphore.release();
            }
        }
    }

    /**
     * @param timeout Реальный тайм-аут, включая потерю соединения.
     */
    @Nonnull
    public BinlogEvent readEvent(Duration timeout) throws InterruptedException {
        Preconditions.checkState(!closed, "Should not be called after close");
        MySQLConnector connector = getSource().getConnector();
        MonotonicTime deadline = NanoTimeClock.now().plus(timeout);
        while (!Thread.currentThread().isInterrupted()) {
            connectOrReconnect(connector, deadline);
            BinlogEvent event = null;
            try {
                event = readEventAndHandleErrors(connector, deadline);
            } catch (IOException exc) {
                handleExceptionWindow(exc);
            }
            if (event != null) {
                lastSeenEventTimestamp = NanoTimeClock.now();
                return event;
            } else if (!NanoTimeClock.now().minus(deadline).isNegative()) {
                throw new RuntimeTimeoutException();
            } else {
                // Сюда можно попасть только при возникновении какой-то ошибки, которую есть шанс починить
                // путём повторного установления соединения.
                forceReconnect = true;
            }
        }
        throw new InterruptedException();
    }

    /**
     * Если ошибки сыпятся слишком часто, то последнюю полученную ошибку бросает выше.
     */
    private void handleExceptionWindow(IOException exc) {
        MonotonicTime now = NanoTimeClock.now();
        exceptionWindow.add(now);
        if (exceptionWindow.size() > exceptionWindowSize) {
            do {
                exceptionWindow.remove();
            } while (exceptionWindow.size() > exceptionWindowSize);
            if (exceptionWindow.peek().plus(exceptionWindowDuration).isAtOrAfter(now)) {
                throw new UncheckedIOException(exc);
            }
        }
    }

    private void connectOrReconnect(MySQLConnector connector, MonotonicTime deadline) throws InterruptedException {
        boolean keepAliveTimedOut = lastSeenEventTimestamp != null
                && lastSeenEventTimestamp.plus(keepAliveTimeout).isAtOrBefore(NanoTimeClock.now());
        boolean shouldEstablishNewConnection = forceReconnect || lastSeenEventTimestamp == null || keepAliveTimedOut;
        if (shouldEstablishNewConnection && sleepUntilConnectTime(deadline)) {
            if (converter.isRawEventSourceSet()) {
                if (keepAliveTimedOut) {
                    logger.warn(
                            "Did not seen any event from {}:{} for {}. Reconnecting, closing old source at gtid set {}.",
                            connector.getHost(), connector.getPort(), keepAliveTimeout,
                            converter.getState().getGtidSet());
                } else {
                    logger.info("Reconnecting to event source for {}:{} at gtid set {}",
                            connector.getHost(), connector.getPort(), converter.getState().getGtidSet());
                }
                Checked.run(converter.getRawEventSource()::close);
            } else {
                logger.info("Connecting to event source for {}:{} at gtid set {}",
                        connector.getHost(), connector.getPort(), converter.getState().getGtidSet());
            }
            converter.attachRawEventSource(Checked.get(() -> new BinlogRawEventServerSource(
                    connector.getHost(),
                    connector.getPort(),
                    connector.getUsername(),
                    connector.getPassword(),
                    getServerId(),
                    converter.getState().getGtidSet(),
                    getMaxBufferedEvents())));
            lastSeenEventTimestamp = NanoTimeClock.now();
            lastConnectWasAt = NanoTimeClock.now();
            forceReconnect = false;
        }
    }

    /**
     * Если сеть становится временно недоступна, то это занимает минимум секунды. Если бы попытки соединения делались в
     * бесконечном цикле, то это привело бы к паре десятков попыток в секунду, что быстро переполнит любой разумный
     * exceptionWindowSize.
     *
     * @param deadline Ждать не дальше этого момента
     * @return true - если deadline не был достигнут, false - если был достигнут
     */
    private boolean sleepUntilConnectTime(MonotonicTime deadline) throws InterruptedException {
        boolean canGoAhead = true;
        if (lastConnectWasAt != null) {
            MonotonicTime nextConnectAttemptAt = lastConnectWasAt.plus(reconnectInterval);
            MonotonicTime sleepUntil;
            if (nextConnectAttemptAt.isBefore(deadline)) {
                sleepUntil = nextConnectAttemptAt;
            } else {
                sleepUntil = deadline;
                canGoAhead = false;
            }
            Duration sleepFor = sleepUntil.minus(NanoTimeClock.now());
            if (!(sleepFor.isNegative() || sleepFor.isZero())) {
                Thread.sleep(sleepFor.toMillis());
            }
        }
        return canGoAhead;
    }

    private BinlogEvent readEventAndHandleErrors(MySQLConnector connector, MonotonicTime deadline)
            throws InterruptedException, IOException {
        MonotonicTime now = NanoTimeClock.now();
        if (now.isAtOrAfter(deadline)) {
            return null;
        }
        Duration deadlineTimeout = deadline.minus(now);
        Duration reconnectTimeout = lastSeenEventTimestamp.plus(keepAliveTimeout).minus(now);
        Duration timeout = deadlineTimeout;
        if (timeout.compareTo(reconnectTimeout) > 0) {
            timeout = reconnectTimeout;
        }
        if (timeout.isNegative()) {
            timeout = Duration.ZERO;
        }

        try {
            BinlogEvent binlogEvent = converter.readEvent(timeout.toNanos(), TimeUnit.NANOSECONDS);
            if (binlogEvent != null) {
                return binlogEvent;
            }
            logger.info("Got EOF from {}:{} at gtid set {}",
                    connector.getHost(), connector.getPort(), converter.getState().getGtidSet());
            // При получении EOF происходит попытка подсоединиться заново.
            return null;
        } catch (EOFException exc) {
            // EOFException - конец бинлога. Это не повод завершать приложение. Может появиться ещё бинлог.
            logger.info("Got EOF from {}:{} at gtid set {}",
                    connector.getHost(), connector.getPort(), converter.getState().getGtidSet());
            return null;
        } catch (TimeoutException exc) {
            // TimeoutException - Это может свидетельствовать либо о превышении дозволенного времени чтения,
            // либо давно не было событий.
            return null;
        } catch (ServerException exc) {
            if (exc.getErrorCode() == 1236 && exc.getMessage().equalsIgnoreCase(""
                    + "A slave with the same server_uuid/server_id"
                    + " as this slave has connected to the master")) {
                throw new BadServerIdException(exc);
            } else {
                throw new IllegalStateException("Got exception while read from "
                        + connector.getHost() + ":" + connector.getPort()
                        + " at gtid set " + converter.getState().getGtidSet(),
                        exc);
            }
        } catch (IOException exc) {
            logger.warn("Got exception while read from {}:{} at gtid set {}",
                    connector.getHost(), connector.getPort(), converter.getState().getGtidSet(), exc);
            throw exc;
        } catch (SQLException | BinlogDDLException exc) {
            throw new Checked.CheckedException(exc);
        }
    }

    public StatefulBinlogSource getSource() {
        return source;
    }

    public BinlogEventConverter getConverter() {
        return converter;
    }

    public int getServerId() {
        return serverId;
    }

    public int getMaxBufferedEvents() {
        return maxBufferedEvents;
    }

    public NamedBinlogState getState() {
        return new NamedBinlogState(source.getName(), converter.getState());
    }

    @Override
    public void close() {
        closed = true;
        try (TmpMySQLServerWithDataDir ignored = schemaReplicaMysql;
             Connection ignored2 = schemaReplicaConn) {
            if (converter.getRawEventSource() != null) {
                converter.getRawEventSource().close();
            }
        } catch (SQLException | IOException exc) {
            throw new Checked.CheckedException(exc);
        }
    }

    public boolean isClosed() {
        return closed;
    }

    public class BadServerIdException extends RuntimeException {
        public BadServerIdException(ServerException exc) {
            super(exc);
        }
    }
}
